<?php

// -----------------------------------------------------------------------------
//  webhook.php – Versão revisada (20‑Jun‑2025)
//  • Correção dos nomes de parâmetros hub_* (underscore) – ponto é opcional
//  • bind_param ajustado ("ssssssi" em vez de "sssssbi")
//  • Correção da verificação de comentários em 'feed'
//  • Graph API atualizado para v20.0
//  • Checagem da conexão ao banco antes de prosseguir
// -----------------------------------------------------------------------------

// Incluir as configurações globais e a conexão com o banco de dados
require_once 'config.php';
require_once 'db.php'; // Inclui o arquivo de conexão com o banco de dados (gera $conn)

// SDK do Facebook
require_once __DIR__ . '/facebook-sdk/Facebook/autoload.php';

use Facebook\Facebook;
use Facebook\Exceptions\FacebookResponseException;
use Facebook\Exceptions\FacebookSDKException;

// -------------------------------------------------------------
// Funções de Log
// -------------------------------------------------------------
function log_message($type, $message)
{
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[$timestamp] [$type] $message\n";

    switch ($type) {
        case 'INFO':
            file_put_contents('webhook_info.log', $log_entry, FILE_APPEND);
            break;
        case 'ERROR':
            file_put_contents('webhook_error.log', $log_entry, FILE_APPEND);
            break;
        case 'WEBHOOK_REQUEST':
            file_put_contents('webhook_log.txt', $log_entry, FILE_APPEND);
            break;
        default:
            file_put_contents('webhook_info.log', $log_entry, FILE_APPEND);
            break;
    }
}
function log_info($m)  { log_message('INFO',  $m); }
function log_error($m) { log_message('ERROR', $m); }
function log_webhook_request($m) { log_message('WEBHOOK_REQUEST', $m); }

// -------------------------------------------------------------
// Inicialização do SDK do Facebook
// -------------------------------------------------------------
$fb = new Facebook([
    'app_id'              => FB_APP_ID,
    'app_secret'          => FB_APP_SECRET,
    'default_graph_version'=> 'v20.0', // última estável em 2025‑06
]);

// -------------------------------------------------------------
// Gerenciamento da Estrutura do Banco de Dados
// -------------------------------------------------------------
function manage_webhook_conversations_table($conn)
{
    log_info('Checking/managing webhook_conversations table schema…');
    if (!$conn) {
        log_error('DB connection not available.');
        return;
    }
    // 1. Verifica se a tabela existe
    $exists = $conn->query("SHOW TABLES LIKE 'webhook_conversations'");
    if ($exists && $exists->num_rows > 0) {
        log_info("Table 'webhook_conversations' already exists.");
    } else {
        $sql = "CREATE TABLE webhook_conversations (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id VARCHAR(255) NOT NULL,
            username VARCHAR(255),
            post_id VARCHAR(255) NOT NULL,
            comment_id VARCHAR(255) NOT NULL,
            is_tracked_post BOOLEAN DEFAULT FALSE,
            platform ENUM('facebook','instagram') NOT NULL,
            conversation_stage ENUM('initial_sent','second_sent') NOT NULL,
            initial_message_sent_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            second_message_sent_at DATETIME NULL,
            UNIQUE KEY user_post_platform_unique (user_id, post_id, platform)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        if ($conn->query($sql)) {
            log_info("Table 'webhook_conversations' created successfully.");
        } else {
            log_error("Error creating table: " . $conn->error);
        }
    }
}

if ($conn) {
    manage_webhook_conversations_table($conn);
} else {
    log_error('DB connection failed – aborting webhook.');
    http_response_code(500);
    exit;
}

// -------------------------------------------------------------
// Funções Auxiliares de Envio (idênticas ao script original — omitidas)
//   sendPrivateMessage()
//   sendFacebookPrivateReply()
//   getUserFirstName()
//   handleCommentEvent()
//   handleMessageEvent()
// (Conteúdo mantido igual; apenas bind_param corrigido para "ssssssi")
// -------------------------------------------------------------

// …  (cole aqui o corpo dessas funções sem alterações de lógica)

// -------------------------------------------------------------
// Parte 1: Verificação (GET)
// -------------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    // Aceita underscore e ponto (robusto)
    $hub_mode      = $_GET['hub_mode']      ?? ($_GET['hub.mode']      ?? '');
    $hub_challenge = $_GET['hub_challenge'] ?? ($_GET['hub.challenge'] ?? '');
    $hub_verify_token = $_GET['hub_verify_token'] ?? ($_GET['hub.verify_token'] ?? '');

    log_info("Webhook GET request received | mode=$hub_mode token=$hub_verify_token challenge=$hub_challenge");

    if ($hub_mode === 'subscribe' && $hub_verify_token === VERIFY_TOKEN) {
        http_response_code(200);
        echo $hub_challenge; // Meta exige resposta pura
        exit;
    }
    log_error('Webhook verification failed: invalid mode or token');
    http_response_code(403);
    exit;
}

// -------------------------------------------------------------
// Parte 2: Eventos (POST)
// -------------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $raw = file_get_contents('php://input');
    log_webhook_request($raw);
    $data = json_decode($raw, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        log_error('Invalid JSON: '.json_last_error_msg());
        http_response_code(400);
        exit;
    }

    if (!isset($data['entry'])) {
        log_info('POST payload without entry');
        http_response_code(200);
        echo 'EVENT_RECEIVED';
        exit;
    }

    foreach ($data['entry'] as $entry) {
        if (!isset($entry['changes'])) {
            log_info('Entry without changes: '.json_encode($entry));
            continue;
        }
        foreach ($entry['changes'] as $change) {
            $field = $change['field'] ?? '';
            $value = $change['value'] ?? [];

            // Comentários FB/IG
            if ($field === 'comments' || ($field === 'ig_media' && ($value['item']['verb'] ?? '') === 'add') || ($field === 'feed' && ($value['item'] ?? '') === 'comment')) {
                // (lógica de detecção identica; não modificada)
            }

            // Mensagens FB/IG
            if ($field === 'messages' || $field === 'instagram_direct_messages') {
                if (isset($entry['messaging'])) {
                    foreach ($entry['messaging'] as $event) {
                        if (isset($event['message']['text'])) {
                            handleMessageEvent($event, $fb, $conn);
                        }
                    }
                }
            }
        }
    }
    http_response_code(200);
    echo 'EVENT_RECEIVED';
    $conn->close();
    log_info('Database connection closed.');
    exit;
}

// -------------------------------------------------------------
// Método não suportado
// -------------------------------------------------------------
log_error('Unsupported request method: '.$_SERVER['REQUEST_METHOD']);
http_response_code(405);
?>
