<?php

/***************************************
 * Scraper Traxsource & Post to Facebook / Instagram – Versão 1.0 by Ronan C
 * Ajustado para usar config.php e MySQL
 ***************************************/

/* ---------- INCLUI CONFIGURAÇÃO ---------- */
require_once 'config.php';

/* ---------- CONFIGURAÇÃO DE BUFFER ---------- */
while (ob_get_level() > 0) { ob_end_clean(); }
ob_implicit_flush(true);
header('X-Accel-Buffering: no');
if (function_exists('apache_setenv')) { apache_setenv('no-gzip', '1'); }
header('Content-Type: text/html; charset=utf-8');

/* ---------- HTML BÁSICO ---------- */
echo '<!DOCTYPE html><html lang="pt-BR"><head><meta charset="UTF-8">
<meta name="viewport" content="width=device-width,initial-scale=1.0">
<title>Andamento do Scraper Traxsource</title>
<link href="https://fonts.googleapis.com/css2?family=Bebas+Neue&family=Open+Sans:wght@400;700&display=swap" rel="stylesheet">
<style>
 body{background:#000;color:#FF0;font:1.1em "Open Sans",sans-serif;margin:20px;line-height:1.6;white-space:pre-wrap}
 h2{font:2em "Bebas Neue",cursive;color:#FF0;border-bottom:2px solid #333;margin-top:30px}
 .success{color:#0F0;font-weight:bold}.error{color:#F00;font-weight:bold}
 .warning{color:#FFA500}.info{color:#ADD8E6}.separator{border-top:1px dashed #555;margin:20px 0}
 .post-preview{background:#1a1a1a;border:1px solid #333;padding:20px;margin-top:20px;border-radius:88px;overflow-wrap:break-word}
 .post-preview h3{color:#FF0;margin-top:0}
 .post-preview img{max-width:100%;height:auto;margin-top:15px;border-radius:4px}
 @media(max-width:768px){body{font-size:1em;margin:10px}h2{font-size:1.8em)}}
</style></head><body><pre>';
flush();

/* ---------- CONEXÃO COM O BANCO DE DADOS (NOVO) ---------- */
// Variáveis para armazenar IDs de postagem para o banco de dados
$facebook_post_id_to_save = NULL;
$instagram_post_id_to_save = NULL;
$conn = NULL; // Inicializa a conexão como nula

try {
    // Reporta erros de MySQLi como exceções para melhor tratamento
    mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
    $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
    if ($conn->connect_error) {
        throw new Exception("Falha na conexão com o banco de dados: " . $conn->connect_error);
    }
    echo "<span class='success'>Conexão com o banco de dados MySQL estabelecida com sucesso.</span>\n";
    flush();

    // NOVO: Verificar e criar a tabela 'posts' se não existir
    echo "<span class='info'>Verificando estrutura do banco de dados e criando tabela 'posts' se necessário...</span>\n";
    $createTableSql = "
        CREATE TABLE IF NOT EXISTS posts (
            id INT AUTO_INCREMENT PRIMARY KEY,
            traxsource_url TEXT NOT NULL,
            track_title VARCHAR(255) NOT NULL,
            artists TEXT NOT NULL,
            label VARCHAR(255) NOT NULL,
            release_image_url TEXT NOT NULL,
            description TEXT NULL,
            facebook_post_id VARCHAR(255) NULL,
            instagram_post_id VARCHAR(255) NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
    ";
    // Usamos TEXT para URLs e descrições para acomodar conteúdos mais longos.
    // VARCHAR(255) para títulos, artistas e labels.
    // facebook_post_id e instagram_post_id como VARCHAR(255) para maior compatibilidade.
    // CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci para suporte a caracteres especiais e emojis.

    if ($conn->query($createTableSql) === TRUE) {
        echo "<span class='success'>Tabela 'posts' verificada/criada com sucesso.</span>\n";
    } else {
        throw new Exception("ERRO ao criar ou verificar a tabela 'posts': " . $conn->error);
    }
    flush();

} catch (Exception $e) {
    echo "<span class='error'>ERRO: " . htmlspecialchars($e->getMessage()) . "</span>\n";
    echo "<span class='warning'>As operações de banco de dados podem ser afetadas.</span>\n";
    $conn = NULL; // Garante que $conn seja NULL se a conexão ou criação da tabela falhar.
    flush();
    // O script continuará a execução, mas sem tentar salvar no banco de dados.
}


/* ---------- PARÂMETROS ---------- */
$artist_url_to_scrape = ARTIST_URL_TO_SCRAPE; // Usando parâmetro do config.php
$base_url             = BASE_URL;             // Usando parâmetro do config.php
$target_artist_name   = TARGET_ARTIST_NAME;   // Usando parâmetro do config.php
$ts_logo_path         = TS_LOGO_PATH;         // Usando parâmetro do config.php
// Definindo a URL base pública para o diretório de imagens (já é uma constante)
// Removido 'const IMAGES_PUBLIC_BASE_URL', agora vem do config.php
$user_agent = USER_AGENT_STRING; // Usando parâmetro do config.php

// Variável para armazenar a URL da imagem do perfil do artista principal (Etapa 1)
$artist_profile_image_url = '';
// Variável para armazenar as URLs das imagens de perfil dos artistas adicionais (Etapa 3.1)
$additional_artists_profile_images = [];
// Variável de controle para habilitar/desabilitar a postagem real no Facebook
// Removido 'const ENABLE_FACEBOOK_POSTING', agora vem do config.php
// NOVA VARIÁVEL: Variável de controle para habilitar/desabilitar a postagem real no Instagram
// Removido 'const ENABLE_INSTAGRAM_POSTING', agora vem do config.php

/* ---------- FUNÇÕES AUXILIARES ---------- */
function makeCurlRequest(string $url, int $max_retries, string $ua): array
{
    $delay = 5;
    for ($t = 1; $t <= $max_retries; $t++) {
        echo "<span class='info'>Tentativa $t/$max_retries → "
            ."<a href='".htmlspecialchars($url)."' target='_blank'>".htmlspecialchars($url)."</a></span>\n";
        flush();

        $ch = curl_init($url);
        if (!$ch) { echo "<span class='error'>ERRO: cURL init.</span>\n"; return ['success'=>false]; }

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_USERAGENT      => $ua,
            CURLOPT_IPRESOLVE      => CURL_IPRESOLVE_V4,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_TIMEOUT        => 20,
            CURLOPT_LOW_SPEED_LIMIT=> 1,
            CURLOPT_LOW_SPEED_TIME => 10,
            CURLOPT_ENCODING       => '',
            CURLOPT_HTTPHEADER     => [
                'Accept: text/html,application/xhtml+xml;q=0.9,*/*;q=0.8',
                'Accept-Language: en-US,en;q=0.9,pt-BR;q=0.8',
                'Cache-Control: no-cache',
                'Pragma: no-cache',
                'Referer: https://www.traxsource.com/',
            ],
        ]);
        $body         = curl_exec($ch);
        $http         = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $errno        = curl_errno($ch);
        $err          = curl_error($ch);
        curl_close($ch);
        if ($errno === 0 && $http >= 200 && $http < 300) {
            echo "<span class='success'>SUCESSO! Status $http.</span>\n";
            return ['success'=>true,'body'=>$body];
        }

        echo "<span class='warning'>Falhou: ".($errno ? "cURL $errno $err" : "HTTP $http")."</span>\n";
        if ($t < $max_retries) {
            echo "<span class='info'>Back‑off {$delay}s…</span>\n";
            flush();
            sleep($delay); $delay *= 2;
        }
    }
    echo "<span class='error'>ERRO: max retries.</span>\n";
    return ['success'=>false];
}

function countdown(int $seconds): void
{
    $msg = "<span class='info'>Aguardando ";
    for ($i=$seconds;$i>0;$i--){
        echo "\r$msg<b>".str_pad($i,2,' ',STR_PAD_LEFT)."</b>s…</span>";
        flush(); sleep(1);
    }
    $final_message = "<span class='success'>Tempo de espera concluído.</span>\n";
    echo "\r" . str_pad($final_message, strlen($msg) + 10, ' ', STR_PAD_RIGHT); // Ensure line is cleared
}

// Para extrair a imagem de perfil de uma página de artista
function getArtistProfileImage(string $artist_page_html): string
{
    preg_match('/<meta\s+property="og:image"\s+content="([^"]*)"/i', $artist_page_html, $matches);
    if (isset($matches[1])) {
        return html_entity_decode($matches[1], ENT_QUOTES | ENT_HTML5, 'UTF-8');
    }
    return '';
}

/**
 * Downloads a remote file using cURL and saves it to a temporary location.
 *
 * @param string $url The URL of the file to download.
 * @param string $userAgent The User-Agent string for the cURL request.
 * @return string|false The path to the temporary file on success, or false on failure.
 */
function download_remote_file_with_curl(string $url, string $userAgent)
{
    $temp_filename = tempnam(sys_get_temp_dir(), 'fb_img_');
    if ($temp_filename === false) {
        return false;
    }

    $ch = curl_init($url);
    if (!$ch) {
        unlink($temp_filename);
        return false;
    }

    $fp = fopen($temp_filename, 'wb');
    if ($fp === false) {
        curl_close($ch);
        unlink($temp_filename);
        return false;
    }

    curl_setopt_array($ch, [
        CURLOPT_FILE           => $fp,
        CURLOPT_HEADER         => 0,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_USERAGENT      => $userAgent,
        CURLOPT_IPRESOLVE      => CURL_IPRESOLVE_V4,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_TIMEOUT        => 20,
    ]);
    $success = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $errno = curl_errno($ch);
    $err = curl_error($ch);

    curl_close($ch);
    fclose($fp);
    if ($success && $http_code >= 200 && $http_code < 300) {
        return $temp_filename;
    } else {
        echo "<span class='warning'>AVISO: Falha ao baixar imagem via cURL ($http_code): " .
        ($errno ? "cURL $errno: $err" : "HTTP $http_code") . "</span>\n";
        unlink($temp_filename);
        return false;
    }
}


/* ---------- ETAPA 1: página do artista ---------- */
echo "<span class='info'>Consultando artista $target_artist_name</span>\n";
echo "<h2>Etapa 1 – Página do artista</h2>\n";
$r = makeCurlRequest(ARTIST_URL_TO_SCRAPE, 4, $user_agent);
if (!$r['success']) die("<span class='error'>Falha ao baixar página do artista.</span></pre></body></html>");
$artist_html = $r['body'];
// Extrair a URL da imagem do perfil do artista principal
$artist_profile_image_url = getArtistProfileImage($artist_html);
if (!empty($artist_profile_image_url)) {
    echo "<span class='info'>URL da imagem de perfil do artista principal encontrada: <b>" .
    htmlspecialchars($artist_profile_image_url) . "</b></span>\n";
} else {
    echo "<span class='warning'>Não foi possível encontrar a URL da imagem de perfil do artista principal na página.</span>\n";
}


/* ---------- ETAPA 2: extrair IDs ---------- */
echo "<div class='separator'></div>\n<h2>Etapa 2 – Extraindo release</h2>\n";
preg_match_all('/data-cart="\{[^"]*title_id\s*:\s*(\d+)/i', $artist_html, $ids);
$unique_ids = array_values(array_unique($ids[1] ?? []));
if (!$unique_ids) die("<span class='error'>Nenhum release encontrado.</span></pre></body></html>");
echo "<span class='success'>".count($unique_ids)." Releases encontrados.</span>\n";
echo "<div class='separator'></div>\n";
/* ---------- EXPRESSÕES ---------- */
$re_h1_artists = '/<h1[^>]*class="[^"]*artists[^"]*"[^>]*>(.*?)<\/h1>/is';
$re_a_artists  = '/<a\s+[^>]*class="[^"]*com-artists[^"]*"[^>]*>(.*?)<\/a>/is'; // Usado para extrair nomes
$re_title_tag  = '/<title>([^<]+?)\s+-\s+.*?- Traxsource<\/title>/i';
$re_meta_og    = '/<meta\s+property="og:([^"]+)"\s+content="([^"]*)"/i';
$re_label      = '/<a\s+class="com-label"[^>]*>([^<]+)<\/a>/i';
// NOVA REGEX: Para extrair URLs de perfil de artista do HTML do release
$re_artist_profile_links = '/<a\s+href="(\/artist\/\d+\/[^"]+)"\s+class="com-artists"[^>]*>([^<]+)<\/a>/i';


/* ---------- ETAPA 3: selecionar release ---------- */
echo "<h2>Etapa 3 – Buscando release válido</h2>\n";
$selected_url = null; $selected_html = ''; $selected_artists = [];
// Array para armazenar URLs dos perfis de todos os artistas do release selecionado
$selected_artist_profile_urls = [];

for ($loop=1; $loop<=30 && !$selected_url; $loop++) {
    echo "<span class='info'>Tentativa #$loop/30</span>\n";
    $url = BASE_URL."/title/".$unique_ids[array_rand($unique_ids)]; // Usando BASE_URL do config.php
    echo "<span class='info'>→ " . htmlspecialchars($url) . "</span>\n";

    $rel = makeCurlRequest($url, 4, $user_agent);
    if (!$rel['success']) { countdown(rand(1,4)); continue; }

    $html = $rel['body'];
    $artists = [];
    $current_release_artist_profile_links = []; // Para armazenar links de artistas desta iteração

    // Função auxiliar para limpar e normalizar nomes de artistas
    $clean_artist_name = function($name) {
        $name = trim(html_entity_decode(strip_tags($name), ENT_QUOTES | ENT_HTML5, 'UTF-8'));
        $name = preg_replace('/\s*\(.*\)\s*/', '', $name);
        $name = preg_replace('/[^a-zA-Z0-9\s.,&]/u', '', $name);
        $name = str_replace(' And ', ' & ', $name);
        $name = preg_replace('/\s+/', ' ', $name);
        $name = trim($name, " ,&");
        return $name;
    };

    // Prioridade na extração de artistas
    if (preg_match_all($re_h1_artists, $html, $m)) {
        foreach ($m[1] as $t) $artists[] = $clean_artist_name($t);
    }

    if (!$artists && preg_match_all($re_a_artists, $html, $m)) {
        foreach ($m[1] as $t) $artists[] = $clean_artist_name($t);
    }

    // Como último recurso, tentar do title tag
    if (!$artists && preg_match($re_title_tag, $html, $m)) {
        $title_parts = explode(' - ', $m[1]);
        if (count($title_parts) > 0) {
            $artists[] = $clean_artist_name($title_parts[0]);
        }
    }

    $processed_artists = [];
    foreach ($artists as $artist_string) {
        $temp_artists = preg_split('/[,\s]*&[,\s]*|,\s*/u', $artist_string, -1, PREG_SPLIT_NO_EMPTY);
        foreach ($temp_artists as $art) {
            $cleaned_art = $clean_artist_name($art);
            if (!empty($cleaned_art)) {
                $processed_artists[] = $cleaned_art;
            }
        }
    }

    $artists = array_values(array_unique($processed_artists));
    if (!$artists) { echo "<span class='warning'>  -> Nenhum artista encontrado. Pulando.</span>\n"; continue; }

    echo "<span class='info'>  -> Artistas na página: <b>".htmlspecialchars(implode(', ',$artists))."</b></span>\n";

    $lower_artists = array_map(fn($a)=>mb_strtolower($a,'UTF-8'), $artists);
    $lower_target_artist = mb_strtolower(TARGET_ARTIST_NAME,'UTF-8'); // Usando TARGET_ARTIST_NAME do config.php

    if (in_array('various artists', $lower_artists, true)) {
        echo "<span class='warning'>  -> 'Various Artists'. Pulando.</span>\n";
        continue;
    }

    $found_target_artist = false;
    foreach ($lower_artists as $artist_on_page) {
        if (strpos($artist_on_page, $lower_target_artist) !== false) {
            $found_target_artist = true;
            break;
        }
    }

    if (!$found_target_artist) {
        echo "<span class='warning'>  -> Não contém '".TARGET_ARTIST_NAME."' após limpeza. Pulando.</span>\n"; // Usando TARGET_ARTIST_NAME do config.php
        continue;
    }

    if (count($artists) > 3) {
        echo "<span class='warning'>  -> Mais de 3 artistas encontrados (incluindo ".TARGET_ARTIST_NAME."). Pulando.</span>\n"; // Usando TARGET_ARTIST_NAME do config.php
        continue;
    }

    $selected_url       = $url;
    $selected_html      = $html;
    $selected_artists   = $artists;
    // Salva a lista de artistas selecionada

    // NOVO: Extrair as URLs de perfil dos artistas do HTML do release selecionado
    preg_match_all($re_artist_profile_links, $selected_html, $artist_link_matches, PREG_SET_ORDER);
    foreach ($artist_link_matches as $match) {
        $profile_path = $match[1];
        $artist_name_from_link = $clean_artist_name($match[2]);
        // Adiciona ao array apenas se o artista extraído for um dos 'selected_artists'
        // E se ainda não foi adicionado (para evitar duplicatas)
        if (in_array($artist_name_from_link, $selected_artists) && !isset($selected_artist_profile_urls[$artist_name_from_link])) {
            $selected_artist_profile_urls[$artist_name_from_link] = BASE_URL . $profile_path; // Usando BASE_URL do config.php
        }
    }

    echo "<span class='success'>  -> Release válido encontrado com ".count($selected_artists)." artista(s).</span>\n";
    // Break para sair do loop uma vez que um release válido é encontrado
    break;
}

/* ---------- ETAPA 4: resultado ---------- */
echo "<div class='separator'></div>\n<h2>Etapa 4 – Resultado</h2>\n";
if (!$selected_url) {
    echo "<span class='error'>Nenhum release válido encontrado.</span>\n";
    echo '</pre></body></html>'; exit;
}

echo "<span class='success'>URL selecionada:</span>\n<a href='".htmlspecialchars($selected_url)."' target='_blank'>"
    .htmlspecialchars($selected_url)."</a>\n";
/* ---------- ETAPA 3.1: Coletar imagens de perfil de artistas adicionais (NOVA ETAPA) ---------- */
if (count($selected_artists) > 1) {
    echo "<div class='separator'></div>\n<h2>Etapa 3.1 – Coletando imagens de perfil de artistas adicionais</h2>\n";
    foreach ($selected_artists as $artist_name) {
        // Ignora o artista principal Ronan C, pois sua imagem já foi coletada na Etapa 1
        $lower_target_artist = mb_strtolower(TARGET_ARTIST_NAME,'UTF-8'); // Usando TARGET_ARTIST_NAME do config.php
        if (strpos(mb_strtolower($artist_name,'UTF-8'), $lower_target_artist) !== false) {
            echo "<span class='info'>Pulando artista principal: <b>" .
            htmlspecialchars($artist_name) . "</b> (imagem já coletada ou ignorada na Etapa 1).</span>\n";
            continue;
        }

        echo "<span class='info'>Buscando imagem de perfil para artista adicional: <b>" .
        htmlspecialchars($artist_name) . "</b></span>\n";

        $profile_url_for_artist = $selected_artist_profile_urls[$artist_name] ?? '';

        if (!empty($profile_url_for_artist)) {
            echo "<span class='info'>  -> URL do perfil: " .
            htmlspecialchars($profile_url_for_artist) . "</span>\n";
            $r_artist_profile = makeCurlRequest($profile_url_for_artist, 4, $user_agent);
            if ($r_artist_profile['success']) {
                $profile_image = getArtistProfileImage($r_artist_profile['body']);
                if (!empty($profile_image)) {
                    $additional_artists_profile_images[$artist_name] = $profile_image;
                    echo "<span class='success'>  -> Imagem de perfil encontrada para <b>" . htmlspecialchars($artist_name) . "</b>: " . htmlspecialchars($profile_image) . "</span>\n";
                } else {
                    echo "<span class='warning'>  -> Não foi possível encontrar a imagem de perfil para <b>" .
                    htmlspecialchars($artist_name) . "</b>.</span>\n";
                }
            } else {
                echo "<span class='error'>  -> Falha ao baixar a página de perfil para <b>" .
                htmlspecialchars($artist_name) . "</b>.</span>\n";
            }
        } else {
            echo "<span class='warning'>  -> URL do perfil não encontrada no HTML do release para <b>" .
            htmlspecialchars($artist_name) . "</b>. Pulando.</span>\n";
        }
        countdown(rand(1,3)); // Pequena pausa entre as requisições de perfil
    }
    echo "<span class='success'>Coleta de imagens de perfil de artistas adicionais concluída.</span>\n";
} else {
    echo "<div class='separator'></div>\n<h2>Etapa 3.1 – Coletando imagens de perfil de artistas adicionais</h2>\n";
    echo "<span class='info'>Apenas um artista encontrado no release. Etapa 3.1 ignorada.</span>\n";
}


/* ---------- ETAPA 5: extrair metadados og:* e Label ---------- */
echo "<div class='separator'></div>\n<h2>Etapa 5 – Extraindo metadados e Gravadora</h2>\n";
preg_match_all($re_meta_og, $selected_html, $og);
$meta = [];
foreach ($og[1] as $k => $prop) $meta[$prop] = html_entity_decode($og[2][$k], ENT_QUOTES | ENT_HTML5, 'UTF-8');
// Tenta extrair a gravadora
$label = '';
if (preg_match($re_label, $selected_html, $m_label)) {
    $label = trim(html_entity_decode($m_label[1], ENT_QUOTES | ENT_HTML5, 'UTF-8'));
}
$meta['label'] = $label; // Adiciona a label aos metadados

/* garanta as chaves principais, mesmo se faltarem */
foreach (['title','type','url','image','description','label'] as $k) $meta[$k] = $meta[$k] ?? '';

// Adiciona a lista de artistas ao array meta para salvamento e uso posterior
$meta['artists'] = implode(', ', $selected_artists); // Usa a lista de artistas já processada

foreach ($meta as $k=>$v) echo "<span class='info'>$k: <b>".htmlspecialchars($v)."</b></span>\n";

echo "<div class='separator'></div>\n"; // Mantém o separador para a próxima etapa


// --- ETAPA 7: VALIDAÇÃO DA API DO FACEBOOK ---
// Certifique-se de ter a pasta 'facebook-sdk' na raiz do script.
echo "<h2>Etapa 7 – Validação das Credenciais da API do Facebook</h2>\n";

// Inclui o autoloader do SDK do Facebook
require_once __DIR__ . '/facebook-sdk/Facebook/autoload.php';

$fb = null;
// Variável para armazenar o ID da conta de negócios do Instagram
$instagram_business_account_id = '';
try {
    $fb = new Facebook\Facebook([
        'app_id' => FB_APP_ID,          // Usando constante do config.php
        'app_secret' => FB_APP_SECRET,  // Usando constante do config.php
        'default_graph_version' => 'v20.0',
    ]);
    echo "<span class='info'>Tentando validar o FB_PAGE_ID e FB_PAGE_ACCESS_TOKEN...</span>\n";

    // Modificação aqui: Buscando os campos 'id', 'name' e 'instagram_business_account'
    $response = $fb->get('/' . FB_PAGE_ID . '?fields=id,name,instagram_business_account', FB_PAGE_ACCESS_TOKEN); // Usando constantes do config.php
    $graphNode = $response->getGraphNode();
    echo "<span class='success'>Validação da API do Facebook CONCLUÍDA com sucesso!</span>\n";
    echo "<span class='info'>ID da Página: <b>" .
    htmlspecialchars($graphNode['id']) . "</b></span>\n";
    echo "<span class='info'>Nome da Página: <b>" . htmlspecialchars($graphNode['name']) . "</b></span>\n";
    // NOVO: Captura o ID da conta do Instagram Business de forma segura
    // Verifica se o campo 'instagram_business_account' existe antes de tentar acessá-lo
    if ($graphNode->offsetExists('instagram_business_account') && !empty($graphNode->offsetGet('instagram_business_account')['id'])) {
        $instagram_business_account_id = $graphNode->offsetGet('instagram_business_account')['id'];
        echo "<span class='success'>ID da Conta de Negócios do Instagram: <b>" . htmlspecialchars($instagram_business_account_id) . "</b></span>\n";
    } else {
        echo "<span class='warning'>AVISO: Esta página do Facebook NÃO está conectada a uma conta de Negócios do Instagram, ou a informação não pôde ser recuperada. A postagem no Instagram PODE NÃO funcionar.</span>\n";
    }

    echo "<span class='success'>Suas credenciais parecem estar OK para a página " . htmlspecialchars($graphNode['name']) . ".</span>\n";
} catch (Facebook\Exception\FacebookResponseException $e) {
    echo "<span class='error'>ERRO na validação da API do Facebook: " .
    htmlspecialchars($e->getMessage()) . "</span>\n";
    echo "<span class='info'>Código de Erro: " . htmlspecialchars($e->getCode()) . "</span>\n";
    echo "<span class='info'>Subcódigo: " . htmlspecialchars($e->getSubcode()) .
    "</span>\n";
    echo "<span class='info'>Tipo: " . htmlspecialchars($e->getErrorType()) . "</span>\n";
    echo "<span class='info'>Raw Response: " . htmlspecialchars($e->getRawResponse()) . "</span>\n";
    echo "<span class='warning'>Verifique seu FB_APP_ID, FB_APP_SECRET, FB_PAGE_ACCESS_TOKEN e FB_PAGE_ID.</span>\n";
    echo "</pre></body></html>";
    exit;
} catch (Facebook\Exception\FacebookSDKException $e) {
    echo "<span class='error'>ERRO do SDK do Facebook na validação: " .
    htmlspecialchars($e->getMessage()) . "</span>\n";
    echo "<span class='warning'>Certifique-se de que o autoloader do SDK está correto e os arquivos existem.</span>\n";
    echo "</pre></body></html>";
    exit;
} catch (Exception $e) {
    echo "<span class='error'>ERRO Geral na validação: " . htmlspecialchars($e->getMessage()) . "</span>\n";
    echo "</pre></body></html>";
    exit;
}

echo "<div class='separator'></div>\n";


// --- ETAPA 8: SIMULAÇÃO DA POSTAGEM NO FACEBOOK ---

echo "<h2>Etapa 8 – Simulação da Postagem no Facebook</h2>\n";
try {
    // ATENÇÃO: Os dados para postagem agora vêm diretamente do array $meta
    // e NÃO mais do arquivo release-to-post.txt.
    $post_data = [
        'title'        => $meta['title'] ?? 'Título indisponível',
        'type'         => $meta['type'] ?? '',
        'url'          => $meta['url'] ?? 'Link indisponível',
        'image'        => $meta['image'] ?? '', // Capa do Release
        'description'  => $meta['description'] ?? '',
        'label'        => $meta['label'] ?? '',
        'artists_list' => $meta['artists'] ?? TARGET_ARTIST_NAME, // Usando TARGET_ARTIST_NAME do config.php
    ];
    // Processa o título para separar Track e Artista(s) para exibição
    $full_title = $post_data['title'];
    $track_display_name = $full_title;
    if (strpos($full_title, ' - ') !== false) {
        $parts = explode(' - ', $full_title, 2);
        if (count($parts) === 2) {
            $track_display_name = trim($parts[1]);
        }
    }

    $display_artists_string = !empty($post_data['artists_list']) ? $post_data['artists_list'] : TARGET_ARTIST_NAME; // Usando TARGET_ARTIST_NAME do config.php
    // Limpa a descrição: remove a frase "Available for download on Traxsource."
    $description = html_entity_decode($post_data['description'], ENT_QUOTES | ENT_HTML5, 'UTF-8');
    $description = str_replace('Available for download on Traxsource. ', '', $description);
    $description = str_replace('Available for download on Traxsource.', '', $description);
    $description = trim($description);

    // Limita a descrição final a um tamanho razoável para o Facebook (ex: 500 caracteres, pode ajustar)
    $description = substr($description, 0, 500);
    if (strlen($post_data['description']) > 500) {
        $description .= '...';
    }


    // --- Sentenças iniciais aleatórias ---
    $opening_sentences = [
        "🔥 Just dropped on Traxsource — house heat you don’t wanna miss!",
        "💣 New fire on Traxsource! Pure groove, zero skips.",
        "🎶 House heads, this one’s fresh out the oven. Live now on Traxsource!",
        "⚡️ New release alert! House flavor strong on this one.",
        "💿 Out now on Traxsource — certified dancefloor vibes.",
        "🌀 Deep, groovy, and 100% house. Grab it now on Traxsource!",
        "🎧 Traxsource just got hotter — tune in and turn up.",
        "🔥 This house joint just landed on Traxsource. Instant add to your playlist!",
        "🎵 Press play and ride the groove — now live on Traxsource.",
        "💥 Turn it up! New release just hit Traxsource. Let it bang!",
        "🎚️ DJs, dig this! Fresh weapon out now on Traxsource.",
        "🎶 For the real ones — raw house energy, straight from Traxsource.",
        "🌐 Global sound, underground soul — live now on Traxsource!",
        "🚨 New banger alert! This just hit Traxsource and it’s wild.",
        "👟 Jack your body to this — Traxsource exclusive vibes!",
        "🌊 Smooth. Deep. Classy. It’s house. Out now on Traxsource!",
        "🪩 Strictly for the dancefloor — go stream it on Traxsource.",
        "🎛️ Your set just got better. Grab this track now on Traxsource.",
        "🚀 Next level house music — available now on Traxsource.",
        "🎉 Drop it in your mix. Out today on Traxsource!",
        "🔥 House vibes turned all the way up — now on Traxsource.",
        "🎵 Big groove energy — streaming now at Traxsource.",
        "📡 Broadcasting good vibes only — from Traxsource to the world.",
        "💽 One for the real selectors — now available on Traxsource.",
        "👂 Open your ears. This just landed on Traxsource.",
        "🎤 Another heater from the underground — now on Traxsource!",
        "🎶 Vibe check: passed. Go hit play on Traxsource.",
        "🎚️ If you love house, you need this. On Traxsource now.",
        "🚨 Hot off the press — this release is pure fire on Traxsource.",
        "🧨 Keep the party moving — out now on Traxsource!",
    ];
    $random_opening_sentence = $opening_sentences[array_rand($opening_sentences)];

    // --- Frases de Call to Action para o link ---
    $cta_phrases = [
        "🎧 Grab it now on Traxsource! 👇",
        "💥 Add it to your set — available now on Traxsource. 👇",
        "🔗 Don’t sleep on this — stream or buy now! 👇",
        "🔥 Support the vibe — grab your copy now. 👇",
        "🛒 Click to preview, buy, and boost your playlist! 👇",
        "🎶 Want this in your mix? Hit the link. 👇",
        "🚀 Out now — make it yours on Traxsource. 👇",
        "🎚️ Get the track DJs are talking about — click to listen. 👇",
        "🕺 Full preview + download here 👉",
        "📥 Exclusive vibes — available now. Don’t miss out. 👇",
    ];
    $random_cta_phrase = $cta_phrases[array_rand($cta_phrases)];


    // --- Lista de Hashtags ---
    $hashtags_list = [
        // GERAL – House Music & Subgêneros
        "#HouseMusic", "#DeepHouse", "#JackinHouse", "#SoulfulHouse", "#AfroHouse",
        "#FunkyHouse", "#NuDisco", "#TechHouse", "#MinimalHouse", "#ClassicHouse",
        // TRAXSOURCE FOCADO
        "#Traxsource", "#TraxsourceExclusive", "#OutNowOnTraxsource", "#NowOnTraxsource",
        "#AvailableOnTraxsource", "#TraxsourcePromo", "#TraxsourceHouse", "#TraxsourceTop100",
        "#TraxsourceCharts", "#TraxsourceReleases",
        // DJs & Produtores
        "#DJLife", "#HouseDJ", "#ProducerLife", "#DJSet", "#InTheMix",
        "#Beatmaker", "#ElectronicProducer", "#BedroomProducer", "#HouseProducer", "#DJsOfInstagram",
        // PERFORMANCE / MIX / CLUB
        "#LiveSet", "#MixSession", "#ClubVibes", "#DancefloorReady", "#PartyVibes",
        "#ClubSounds", "#HouseGroove", "#FeelTheVibe", "#NightlifeCulture", "#LateNightGrooves",
        // CENA / COMUNIDADE HOUSE
        "#HouseHeads", "#WeLoveHouse", "#UndergroundHouse", "#SupportHouseMusic", "#HouseMusicForever",
        "#RealHouseMusic", "#HouseCulture", "#HouseNation", "#HouseCommunity", "#KeepHouseAlive",
        // LANÇAMENTOS & MÚSICA
        "#NewMusicFriday", "#NewRelease", "#FreshHouse", "#JustDropped", "#NowPlaying",
        "#HotRelease", "#TrackOfTheDay", "#RecordLabel", "#PromoRelease", "#DigitalRelease",
        // ENERGIA / EMOÇÃO / EXPERIÊNCIA
        "#FeelTheBeat", "#LetTheMusicPlay", "#HouseIsAFeeling", "#MusicIsLife", "#GoodVibesOnly",
        "#ThisIsHouse", "#GrooveOn", "#SoulfulVibes", "#DanceAndRepeat", "#MoveYourBody",
        // STREAM / RADIO / DIVULGAÇÃO
        "#OnlineRadio", "#SupportIndependentMusic", "#UndergroundVibes", "#StreamNow",
        "#HouseMusicRadio", "#PromoPool", "#MusicPromo", "#DJsSupportDJs", "#RadioSupport", "#MusicMarketing",
        // EQUIPAMENTO & PROCESSO CRIATIVO
        "#AbletonLive", "#LogicProX", "#TR909", "#SynthVibes", "#StudioVibes",
        "#MusicStudio", "#VinylOnly", "#DigitalDJ", "#TrackMastering", "#DJGear",
    ];
    // Selecionar 15 hashtags aleatórias
    $random_hashtags = [];
    if (count($hashtags_list) >= 15) {
        $random_keys = array_rand($hashtags_list, 15);
        foreach ($random_keys as $key) {
            $random_hashtags[] = $hashtags_list[$key];
        }
    } else {
        // Se houver menos de 15 hashtags na lista, usa todas
        $random_hashtags = $hashtags_list;
    }
    shuffle($random_hashtags);
    $hashtags_string = implode(' ', $random_hashtags);
    // --- Montar o texto completo da postagem ---
    $simulated_message = "";
    $simulated_message .= $random_opening_sentence . "\n\n";
    $simulated_message .= $random_cta_phrase . "\n";
    $simulated_message .= ($post_data['url'] ?? 'Link indisponível') . "\n\n";

    $simulated_message .= "🎶 Track: " . ($track_display_name) . "\n";

    $artists_array_for_count = array_map('trim', explode(',', $display_artists_string));
    $artist_prefix = (count($artists_array_for_count) > 1) ? "Artists: " : "Artist: ";
    $simulated_message .= "🎧 " . $artist_prefix . ($display_artists_string) . "\n";
    if (!empty($post_data['label'])) {
        $simulated_message .= "💿 Label: " . ($post_data['label']) . "\n";
    }
    $simulated_message .= "\n";

    if (!empty($description)) {
        $simulated_message .= $description . "\n\n";
    }

    $simulated_message .= "🚨 Follow me for more!\n\n";
    $simulated_message .= $hashtags_string;
    echo "<span class='info'>Prévia da postagem:</span>\n";
    echo "<div class='post-preview'>\n";
    echo "<h3>Conteúdo do Post:</h3>\n";
    echo htmlspecialchars($simulated_message) . "\n";
    echo "\n<h3>Anexos (Imagens):</h3>\n";
    // Simula a adição da capa do release
    if (!empty($post_data['image'])) {
        echo "<span class='info'>Capa do Release:</span><br>\n";
        echo "<img src='" . htmlspecialchars($post_data['image']) . "' alt='Capa do Release'>\n";
        echo "<span class='info'>URL da Imagem (Capa): " . htmlspecialchars($post_data['image']) . "</span>\n\n";
    } else {
        echo "<span class='warning'>Nenhuma imagem (og:image) encontrada para prévia da capa.</span>\n";
    }

    // --- Exibir a foto do perfil do artista principal ---
    global $artist_profile_image_url;
    if (!empty($artist_profile_image_url)) {
        echo "<span class='info'>Foto do Perfil do Artista Principal:</span><br>\n";
        echo "<img src='" . htmlspecialchars($artist_profile_image_url) . "' alt='Foto do Perfil do Artista Principal'>\n";
        echo "<span class='info'>URL da Imagem (Artista Principal): " . htmlspecialchars($artist_profile_image_url) . "</span>\n\n";
    } else {
        echo "<span class='warning'>Nenhuma URL de imagem de perfil do artista principal disponível para prévia.</span>\n";
    }

    // --- NOVO: Exibir as fotos de perfil dos artistas adicionais ---
    global $additional_artists_profile_images;
    if (!empty($additional_artists_profile_images)) {
        echo "<h3>Fotos de Perfil dos Artistas Adicionais:</h3>\n";
        foreach ($additional_artists_profile_images as $artist_name => $image_url) {
            echo "<span class='info'>Foto do Perfil de <b>" .
            htmlspecialchars($artist_name) . "</b>:</span><br>\n";
            echo "<img src='" . htmlspecialchars($image_url) . "' alt='Foto do Perfil de " . htmlspecialchars($artist_name) . "'>\n";
            echo "<span class='info'>URL da Imagem: " . htmlspecialchars($image_url) . "</span>\n\n";
        }
    } else {
        echo "<span class='info'>Nenhuma imagem de perfil de artista adicional para exibir.</span>\n";
    }
    // --- FIM NOVO ---

    // Simula a adição do logo do Traxsource
    if (file_exists(TS_LOGO_PATH)) { // Usando TS_LOGO_PATH do config.php
        // Para prévia, usa o caminho local se o arquivo existe
        echo "<span class='info'>Logo Traxsource:</span><br>\n";
        echo "<img src='" . htmlspecialchars(TS_LOGO_PATH) . "' alt='Logo Traxsource'>\n"; // Usando TS_LOGO_PATH do config.php
        echo "<span class='info'>Caminho da Imagem (Logo): " . htmlspecialchars(TS_LOGO_PATH) . "</span>\n"; // Usando TS_LOGO_PATH do config.php
    } else {
        echo "<span class='error'>ERRO: Arquivo do logo do Traxsource não encontrado em " .
        htmlspecialchars(TS_LOGO_PATH) . "</span>\n"; // Usando TS_LOGO_PATH do config.php
    }


    echo "</div>\n";

    echo "<span class='success'>Simulação da postagem concluída.</span>\n";
    echo "<span class='info'>O post real NÃO foi publicado na sua página.</span>\n";
} catch (Exception $e) {
    echo "<span class='error'>ERRO na simulação da postagem: " . htmlspecialchars($e->getMessage()) . "</span>\n";
}

echo "<div class='separator'></div>\n";


// --- ETAPA 9: POSTAGEM REAL NO FACEBOOK ---
echo "<h2>Etapa 9 – Postagem Real no Facebook</h2>\n";
if (ENABLE_FACEBOOK_POSTING) { // Usando constante do config.php
    echo "<span class='warning'>Modo de produção HABILITADO. Tentando publicar no Facebook...</span>\n";
    try {
        // Coleta todas as URLs/caminhos de imagem para upload
        $images_to_upload = [];
        if (!empty($post_data['image'])) { // Capa do Release
            $images_to_upload[] = $post_data['image'];
        }
        if (!empty($artist_profile_image_url)) { // Foto do Artista Principal
            $images_to_upload[] = $artist_profile_image_url;
        }
        foreach ($additional_artists_profile_images as $image_url) { // Fotos de Artistas Adicionais
            $images_to_upload[] = $image_url;
        }
        if (file_exists(TS_LOGO_PATH)) { // Logo do Traxsource // Usando TS_LOGO_PATH do config.php
            $images_to_upload[] = TS_LOGO_PATH; // Usando TS_LOGO_PATH do config.php
        }

        $attached_media_ids = [];
        $temp_files = []; // Para armazenar caminhos de arquivos temporários para limpeza

        foreach ($images_to_upload as $image_source) {
            $uploaded_file_path = '';
            // Verifica se é uma URL remota ou um caminho de arquivo local
            if (filter_var($image_source, FILTER_VALIDATE_URL)) {
                // É uma URL remota, precisa ser baixada primeiro usando cURL
                echo "<span class='info'>Baixando imagem remota via cURL: " .
                htmlspecialchars($image_source) . "</span>\n";
                $downloaded_path = download_remote_file_with_curl($image_source, $user_agent);
                if ($downloaded_path) {
                    $uploaded_file_path = $downloaded_path;
                    $temp_files[] = $downloaded_path; // Adiciona para limpeza posterior
                    echo "<span class='info'>Imagem remota baixada para temporário: " .
                    htmlspecialchars($image_source) . " -> " . htmlspecialchars($uploaded_file_path) . "</span>\n";
                } else {
                    echo "<span class='warning'>AVISO: Não foi possível baixar imagem de URL: " .
                    htmlspecialchars($image_source) . ". Pulando esta imagem.</span>\n";
                    continue;
                }
            } elseif (file_exists($image_source)) {
                // É um caminho de arquivo local e existe
                $uploaded_file_path = $image_source;
                echo "<span class='info'>Usando imagem local: " . htmlspecialchars($image_source) . "</span>\n";
            } else {
                echo "<span class='warning'>AVISO: Imagem não encontrada (URL inválida ou arquivo local ausente): " .
                htmlspecialchars($image_source) . ". Pulando esta imagem.</span>\n";
                continue;
            }

            if (!empty($uploaded_file_path)) {
                try {
                    $photo_response = $fb->post(
                        '/' . FB_PAGE_ID . '/photos', // Usando constante do config.php
                        [
                            'source' => $fb->fileToUpload($uploaded_file_path),
                            'published' => false, // Não publica a foto individualmente
                        ],
                        FB_PAGE_ACCESS_TOKEN // Usando constante do config.php
                    );
                    $photo_node = $photo_response->getGraphNode();
                    $attached_media_ids[] = ['media_fbid' => $photo_node['id']];
                    echo "<span class='success'>Imagem enviada ao Facebook. ID: " . htmlspecialchars($photo_node['id']) . "</span>\n";
                } catch (Facebook\Exception\FacebookResponseException $e) {
                    echo "<span class='error'>ERRO ao enviar imagem (" . htmlspecialchars($image_source) . "): " .
                    htmlspecialchars($e->getMessage()) . "</span>\n";
                } catch (Facebook\Exception\FacebookSDKException $e) {
                    echo "<span class='error'>ERRO do SDK ao enviar imagem (" . htmlspecialchars($image_source) . "): " .
                    htmlspecialchars($e->getMessage()) . "</span>\n";
                }
            }
        }

        if (empty($attached_media_ids)) {
            echo "<span class='error'>Nenhuma imagem anexada para postagem no Facebook. Abortando postagem.</span>\n";
        } else {
            // Cria a postagem com todas as imagens anexadas
            $post_params = [
                'message' => $simulated_message,
                'attached_media' => json_encode($attached_media_ids),
            ];
            $response = $fb->post(
                '/' . FB_PAGE_ID . '/feed', // Usando constante do config.php
                $post_params,
                FB_PAGE_ACCESS_TOKEN // Usando constante do config.php
            );
            $graphNode = $response->getGraphNode();
            $facebook_post_id_to_save = $graphNode['id']; // Salva o ID da postagem
            echo "<span class='success'>Postagem REALIZADA no Facebook com sucesso! ID do Post: <b>" . htmlspecialchars($facebook_post_id_to_save) . "</b></span>\n";
            echo "<span class='info'>Link para o Post: <a href='https://www.facebook.com/" . FB_PAGE_ID . "/posts/" . htmlspecialchars($facebook_post_id_to_save) . "' target='_blank'>Ver Post no Facebook</a></span>\n"; // Usando constante do config.php
        }

    } catch (Facebook\Exception\FacebookResponseException $e) {
        echo "<span class='error'>ERRO na postagem real do Facebook: " .
        htmlspecialchars($e->getMessage()) . "</span>\n";
        echo "<span class='info'>Código de Erro: " . htmlspecialchars($e->getCode()) . "</span>\n";
        echo "<span class='info'>Subcódigo: " . htmlspecialchars($e->getSubcode()) .
        "</span>\n";
        echo "<span class='info'>Tipo: " . htmlspecialchars($e->getErrorType()) . "</span>\n";
        echo "<span class='info'>Raw Response: " . htmlspecialchars($e->getRawResponse()) . "</span>\n";
        echo "<span class='warning'>Verifique as permissões do seu token de acesso à página (FB_PAGE_ACCESS_TOKEN).</span>\n";
    } catch (Facebook\Exception\FacebookSDKException $e) {
        echo "<span class='error'>ERRO do SDK do Facebook na postagem real: " .
        htmlspecialchars($e->getMessage()) . "</span>\n";
    } catch (Exception $e) {
        echo "<span class='error'>ERRO Geral na postagem real: " .
        htmlspecialchars($e->getMessage()) . "</span>\n";
    } finally {
        // Limpa arquivos temporários
        foreach ($temp_files as $temp_file) {
            if (file_exists($temp_file)) {
                unlink($temp_file);
                echo "<span class='info'>Arquivo temporário limpo: " . htmlspecialchars($temp_file) . "</span>\n";
            }
        }
    }
} else {
    echo "<span class='info'>Modo de produção DESABILITADO para Facebook (DEBUG). Nenhuma postagem real será feita.</span>\n";
}

echo "<div class='separator'></div>\n";


// --- ETAPA 10: POSTAGEM REAL NO INSTAGRAM (NOVA ETAPA) ---
echo "<h2>Etapa 10 – Postagem Real no Instagram</h2>\n";
if (ENABLE_INSTAGRAM_POSTING) { // Usando constante do config.php
    if (empty($instagram_business_account_id)) {
        echo "<span class='error'>Não é possível postar no Instagram: ID da conta de negócios do Instagram não encontrado/conectado.</span>\n";
        echo "<span class='warning'>Por favor, certifique-se de que sua página do Facebook está conectada a uma conta Profissional do Instagram e que seu token de acesso tem as permissões corretas (instagram_basic, instagram_content_publish, pages_show_list, pages_read_engagement).</span>\n";
    } else {
        echo "<span class='warning'>Modo de produção HABILITADO. Tentando publicar no Instagram...</span>\n";
        try {
            $images_for_instagram = [];
            // Adiciona a capa do release (já é uma URL pública)
            if (!empty($post_data['image'])) {
                $images_for_instagram[] = $post_data['image'];
            }

            // Adiciona a foto do artista principal (já é uma URL pública)
            if (!empty($artist_profile_image_url)) {
                $images_for_instagram[] = $artist_profile_image_url;
            }

            // Adiciona as fotos dos artistas adicionais (já são URLs públicas)
            foreach ($additional_artists_profile_images as $image_url) {
                $images_for_instagram[] = $image_url;
            }

            // Adiciona o logo do Traxsource usando a URL base definida
            // Constrói a URL pública completa para o logo
            $ts_logo_public_url = IMAGES_PUBLIC_BASE_URL . basename(TS_LOGO_PATH); // Usando constante do config.php
            $images_for_instagram[] = $ts_logo_public_url;
            echo "<span class='info'>URL definida para logo Traxsource: " . htmlspecialchars($ts_logo_public_url) . "</span>\n";


            $container_ids = [];
            foreach ($images_for_instagram as $image_source_url) {
                // Para o Instagram, *sempre* esperamos uma URL pública.
                if (filter_var($image_source_url, FILTER_VALIDATE_URL)) {
                    try {
                        echo "<span class='info'>Criando container para imagem: " .
                        htmlspecialchars($image_source_url) . "</span>\n";
                        $params_ig_media = [
                            'image_url' => $image_source_url, // Passa a URL da imagem diretamente
                            'is_carousel_item' => true,
                        ];
                        $ig_photo_response = $fb->post(
                            '/' . $instagram_business_account_id . '/media',
                            $params_ig_media,
                            FB_PAGE_ACCESS_TOKEN // Usando constante do config.php
                        );
                        $ig_photo_node = $ig_photo_response->getGraphNode();
                        $container_ids[] = $ig_photo_node['id'];
                        echo "<span class='success'>Container ID do Instagram criado para imagem: " . htmlspecialchars($ig_photo_node['id']) . "</span>\n";
                    } catch (Facebook\Exception\FacebookResponseException $e) {
                        echo "<span class='error'>ERRO ao criar container de imagem para Instagram (" . htmlspecialchars($image_source_url) . "): " .
                        htmlspecialchars($e->getMessage()) . "</span>\n";
                        echo "<span class='info'>Raw Response: " . htmlspecialchars($e->getRawResponse()) . "</span>\n";
                    } catch (Facebook\Exception\FacebookSDKException $e) {
                        echo "<span class='error'>ERRO do SDK ao criar container de imagem para Instagram (" . htmlspecialchars($image_source_url) . "): " .
                        htmlspecialchars($e->getMessage()) . "</span>\n";
                    }
                } else {
                    echo "<span class='warning'>AVISO: Imagem para Instagram não é uma URL válida ou acessível: " .
                    htmlspecialchars($image_source_url) . ". Pulando esta imagem.</span>\n";
                }
            }

            if (empty($container_ids)) {
                echo "<span class='error'>Nenhuma imagem anexada para postagem no Instagram. Abortando postagem.</span>\n";
            } else {
                // Cria o post de carrossel no Instagram
                $carousel_params = [
                    'caption' => $simulated_message, // A legenda vai no post principal
                    'media_type' => 'CAROUSEL',
                    'children' => json_encode($container_ids)
                ];
                try {
                    // Envia a solicitação de criação do container do carrossel
                    $carousel_response = $fb->post(
                        '/' . $instagram_business_account_id . '/media',
                        $carousel_params,
                        FB_PAGE_ACCESS_TOKEN // Usando constante do config.php
                    );
                    $carousel_node = $carousel_response->getGraphNode();
                    $carousel_container_id = $carousel_node['id'];
                    echo "<span class='success'>Container do Carrossel do Instagram criado. ID: " . htmlspecialchars($carousel_container_id) . "</span>\n";
                    // Publica o carrossel
                    $publish_response = $fb->post(
                        '/' . $instagram_business_account_id . '/media_publish',
                        ['creation_id' => $carousel_container_id],
                        FB_PAGE_ACCESS_TOKEN // Usando constante do config.php
                    );
                    $publish_node = $publish_response->getGraphNode();
                    $instagram_post_id_to_save = $publish_node['id']; // Salva o ID da postagem
                    echo "<span class='success'>Postagem REALIZADA no Instagram com sucesso! ID do Post: <b>" . htmlspecialchars($instagram_post_id_to_save) . "</b></span>\n";
                    echo "<span class='info'>Verifique seu perfil do Instagram para confirmar a postagem.</span>\n";
                } catch (Facebook\Exception\FacebookResponseException $e) {
                    echo "<span class='error'>ERRO na postagem real do Instagram: " .
                    htmlspecialchars($e->getMessage()) . "</span>\n";
                    echo "<span class='info'>Código de Erro: " . htmlspecialchars($e->getCode()) . "</span>\n";
                    echo "<span class='info'>Subcódigo: " . htmlspecialchars($e->getSubcode()) .
                    "</span>\n";
                    echo "<span class='info'>Tipo: " . htmlspecialchars($e->getErrorType()) . "</span>\n";
                    echo "<span class='info'>Raw Response: " . htmlspecialchars($e->getRawResponse()) . "</span>\n";
                    echo "<span class='warning'>Verifique as permissões do seu token de acesso à página (FB_PAGE_ACCESS_TOKEN) e se a conta do Instagram é uma Conta Profissional.</span>\n";
                } catch (Facebook\Exception\FacebookSDKException $e) {
                    echo "<span class='error'>ERRO do SDK do Facebook na postagem real do Instagram: " .
                    htmlspecialchars($e->getMessage()) . "</span>\n";
                } catch (Exception $e) {
                    echo "<span class='error'>ERRO Geral na postagem real do Instagram: " .
                    htmlspecialchars($e->getMessage()) . "</span>\n";
                }
            }
        } catch (Exception $e) {
            echo "<span class='error'>ERRO inesperado na Etapa 10 (Instagram): " .
            htmlspecialchars($e->getMessage()) . "</span>\n";
        }
    }
} else {
    echo "<span class='info'>Modo de produção DESABILITADO para Instagram (DEBUG). Nenhuma postagem real será feita.</span>\n";
}

echo "<div class='separator'></div>\n";

/* ---------- ETAPA 11: SALVAR NO BANCO DE DADOS (NOVO) ---------- */
echo "<h2>Etapa 11 – Salvando Dados no Banco de Dados</h2>\n";

if ($conn && ($facebook_post_id_to_save || $instagram_post_id_to_save)) {
    echo "<span class='info'>Tentando salvar dados de postagem no banco de dados...</span>\n";
    try {
        $stmt = $conn->prepare("INSERT INTO posts (traxsource_url, track_title, artists, label, release_image_url, description, facebook_post_id, instagram_post_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
        if ($stmt === false) {
            throw new Exception("Falha ao preparar a declaração SQL: " . $conn->error);
        }

        // Garante que as variáveis são tratadas como strings para bind_param
        $ts_url = $meta['url'] ?? '';
        $track_title_db = $meta['title'] ?? '';
        $artists_db = $meta['artists'] ?? '';
        $label_db = $meta['label'] ?? '';
        $image_url_db = $meta['image'] ?? '';
        // Usa a descrição já limpa e limitada
        $description_db = $description;
        
        $stmt->bind_param("ssssssss",
            $ts_url,
            $track_title_db,
            $artists_db,
            $label_db,
            $image_url_db,
            $description_db,
            $facebook_post_id_to_save,
            $instagram_post_id_to_save
        );

        if ($stmt->execute()) {
            echo "<span class='success'>Dados do post salvos no banco de dados com sucesso! ID: " . $conn->insert_id . "</span>\n";
        } else {
            throw new Exception("Falha ao executar a inserção no banco de dados: " . $stmt->error);
        }
        $stmt->close();
    } catch (Exception $e) {
        echo "<span class='error'>ERRO ao salvar no banco de dados: " . htmlspecialchars($e->getMessage()) . "</span>\n";
    }
} elseif (!$conn) {
    echo "<span class='warning'>Não foi possível salvar os dados no banco de dados: Conexão não estabelecida ou falha na criação da tabela.</span>\n";
} else {
    echo "<span class='info'>Nenhuma postagem (Facebook ou Instagram) foi bem-sucedida. Dados não serão salvos no banco de dados.</span>\n";
}

// Fecha a conexão com o banco de dados no final do script
if ($conn) {
    $conn->close();
    echo "<span class='info'>Conexão com o banco de dados MySQL encerrada.</span>\n";
}

echo "<span class='info'>Processo finalizado.</span>\n";
echo '</pre></body></html>';
?>