<?php

/***************************************
 * Engagement Script for Facebook & Instagram Comments – Versão 1.4 by Ronan C
 * Automaticamente responde a comentários em posts do Facebook e Instagram.
 * Inclui modo DEBUG para validação sem postar, com prévia da resposta.
 * Agora com menção dinâmica de perfil social e ícone de gratidão.
 * Correção: TypeError em isPageComment para GraphNode.
 * Ajuste na ordem da frase de resposta.
 ***************************************/

/* ---------- INCLUI CONFIGURAÇÃO ---------- */
require_once 'config.php';

/* ---------- CONFIGURAÇÃO DE BUFFER ---------- */
while (ob_get_level() > 0) { ob_end_clean(); }
ob_implicit_flush(true);
header('X-Accel-Buffering: no');
if (function_exists('apache_setenv')) { apache_setenv('no-gzip', '1'); }
header('Content-Type: text/html; charset=utf-8');

/* ---------- HTML BÁSICO ---------- */
echo '<!DOCTYPE html><html lang="pt-BR"><head><meta charset="UTF-8">
<meta name="viewport" content="width=device-width,initial-scale=1.0">
<title>Andamento do Engagement Script</title>
<link href="https://fonts.googleapis.com/css2?family=Bebas+Neue&family=Open+Sans:wght@400;700&display=swap" rel="stylesheet">
<style>
 body{background:#000;color:#FF0;font:1.1em "Open Sans",sans-serif;margin:20px;line-height:1.6;white-space:pre-wrap}
 h2{font:2em "Bebas Neue",cursive;color:#FF0;border-bottom:2px solid #333;margin-top:30px}
 .success{color:#0F0;font-weight:bold}.error{color:#F00;font-weight:bold}
 .warning{color:#FFA500}.info{color:#ADD8E6}.separator{border-top:1px dashed #555;margin:20px 0}
 .comment-box{background:#1a1a1a;border:1px solid #333;padding:15px;margin-top:10px;border-radius:8px;overflow-wrap:break-word}
 .comment-box p{margin:5px 0}
 .comment-box .author{font-weight:bold;color:#FFF}.comment-box .message{color:#EEE}
 .simulated-message{background:#333;padding:8px;margin-top:5px;border-radius:5px;color:#FFF;font-style:italic;}
 @media(max-width:768px){body{font-size:1em;margin:10px}h2{font-size:1.8em)}}
</style></head><body><pre>';
flush();

/* ---------- CONEXÃO COM O BANCO DE DADOS ---------- */
$conn = NULL; // Inicializa a conexão como nula
try {
    mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
    $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
    if ($conn->connect_error) {
        throw new Exception("Falha na conexão com o banco de dados: " . $conn->connect_error);
    }
    echo "<span class='success'>Conexão com o banco de dados MySQL estabelecida com sucesso.</span>\n";
    flush();

    // Criar tabela 'replied_comments' se não existir
    echo "<span class='info'>Verificando estrutura do banco de dados e criando tabela 'replied_comments' se necessário...</span>\n";
    $createTableSql = "
        CREATE TABLE IF NOT EXISTS replied_comments (
            id INT AUTO_INCREMENT PRIMARY KEY,
            platform VARCHAR(10) NOT NULL, -- 'facebook' ou 'instagram'
            post_id VARCHAR(255) NOT NULL,
            comment_id VARCHAR(255) NOT NULL,
            replied_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY (platform, comment_id)
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
    ";
    if ($conn->query($createTableSql) === TRUE) {
        echo "<span class='success'>Tabela 'replied_comments' verificada/criada com sucesso.</span>\n";
    } else {
        throw new Exception("ERRO ao criar ou verificar a tabela 'replied_comments': " . $conn->error);
    }
    flush();

} catch (Exception $e) {
    echo "<span class='error'>ERRO: " . htmlspecialchars($e->getMessage()) . "</span>\n";
    echo "<span class='warning'>As operações de banco de dados podem ser afetadas.</span>\n";
    $conn = NULL;
    flush();
    // O script continuará, mas sem rastreamento de respostas.
}

/* ---------- FUNÇÕES AUXILIARES ---------- */
function countdown(int $seconds): void
{
    $msg = "<span class='info'>Aguardando ";
    for ($i=$seconds;$i>0;$i--){
        echo "\r$msg<b>".str_pad($i,2,' ',STR_PAD_LEFT)."</b>s…</span>";
        flush(); sleep(1);
    }
    $final_message = "<span class='success'>Tempo de espera concluído.</span>\n";
    echo "\r" . str_pad($final_message, strlen($msg) + 10, ' ', STR_PAD_RIGHT);
}

// Lista de frases para resposta
$reply_phrases = [
    "Thanks for the support! ❤️",
    "Appreciate it! ❤️",
    "Much love! ❤️",
    "Thank you for the love! ❤️🎶",
    "Grateful for your support! ❤️",
    "Big thanks! ❤️✨",
    "Means a lot, thank you! ❤️",
    "Thanks for vibing with us! ❤️🎵",
    "Thanks for showing love! ❤️",
    "Love and respect! ❤️🙏",
    "Thanks for listening! ❤️🎧",
    "Always grateful! ❤️",
    "Thank you, truly appreciated! ❤️",
    "Happy you liked it! ❤️🔥",
    "That means a lot, thank you! ❤️",
    "Thanks for rocking with us! ❤️🎛️",
    "Heartfelt thanks! ❤️",
    "Thanks for the energy! ❤️⚡",
    "Your support fuels us! ❤️",
    "Thanks for sharing the vibe! ❤️🪩"
];

/**
 * Seleciona uma frase de resposta aleatória e adiciona o handle social e hashtags.
 * @param string $platform 'facebook' ou 'instagram'.
 * @return string A frase de resposta completa.
 */
function getRandomReplyPhrase(string $platform): string {
    global $reply_phrases;
    $selected_phrase = $reply_phrases[array_rand($reply_phrases)];

    $social_handle = '';
    if ($platform === 'facebook' && defined('FACEBOOK_PROFILE_HANDLE')) {
        $social_handle = FACEBOOK_PROFILE_HANDLE;
    } elseif ($platform === 'instagram' && defined('INSTAGRAM_PROFILE_HANDLE')) {
        $social_handle = INSTAGRAM_PROFILE_HANDLE;
    }

    $final_message = $selected_phrase;

    if (!empty($social_handle)) {
        // Adiciona "Follow me for more [handle] ✨"
        $final_message .= " All my music in one place: https://linktr.ee/djronanc follow me for more " . $social_handle . " ✨";
    }

    // Adiciona duas quebras de linha e as hashtags
    $final_message .= "\n\n#housemusic #ronanc";

    return $final_message;
}

/**
 * Verifica se um comentário já foi respondido por este script (usando a tabela replied_comments).
 * @param mysqli $conn Conexão com o banco de dados.
 * @param string $platform 'facebook' ou 'instagram'.
 * @param string $comment_id ID do comentário.
 * @return bool True se já foi respondido por este script, false caso contrário.
 */
function isCommentReplied(mysqli $conn, string $platform, string $comment_id): bool {
    if (!$conn) return false; // Se a conexão falhou, não podemos verificar
    try {
        $stmt = $conn->prepare("SELECT 1 FROM replied_comments WHERE platform = ? AND comment_id = ? LIMIT 1");
        $stmt->bind_param("ss", $platform, $comment_id);
        $stmt->execute();
        $stmt->store_result();
        $found = $stmt->num_rows > 0;
        $stmt->close();
        return $found;
    } catch (Exception $e) {
        echo "<span class='error'>ERRO ao verificar comentário respondido no DB: " . htmlspecialchars($e->getMessage()) . "</span>\n";
        return false; // Em caso de erro, assume que não foi respondido para tentar novamente
    }
}

/**
 * Marca um comentário como respondido por este script.
 * @param mysqli $conn Conexão com o banco de dados.
 * @param string $platform 'facebook' ou 'instagram'.
 * @param string $post_id ID do post ao qual o comentário pertence.
 * @param string $comment_id ID do comentário.
 * @return bool True em caso de sucesso, false em caso de falha.
 */
function markCommentAsReplied(mysqli $conn, string $platform, string $post_id, string $comment_id): bool {
    if (!$conn) return false;
    try {
        $stmt = $conn->prepare("INSERT INTO replied_comments (platform, post_id, comment_id) VALUES (?, ?, ?)");
        $stmt->bind_param("sss", $platform, $post_id, $comment_id);
        $success = $stmt->execute();
        $stmt->close();
        return $success;
    } catch (Exception $e) {
        // Ignorar erro de chave duplicada, significa que já foi inserido
        if ($e->getCode() === 1062) { // MySQL error code for duplicate entry
            return true; // Considera como sucesso, pois já estava lá
        }
        echo "<span class='error'>ERRO ao marcar comentário como respondido no DB: " . htmlspecialchars($e->getMessage()) . "</span>\n";
        return false;
    }
}

/**
 * Verifica se um objeto 'from' (usuário ou página) é a própria página/conta de negócios.
 * @param array $from_data Dados do remetente do comentário/reply (ex: $comment['from']->asArray()).
 * @param string $page_or_business_id_to_check O ID da página do Facebook ou conta de negócios do Instagram.
 * @return bool True se for a própria página, false caso contrário.
 */
function isPageComment(array $from_data, string $page_or_business_id_to_check): bool {
    return isset($from_data['id']) && $from_data['id'] === $page_or_business_id_to_check;
}


/* ---------- ETAPA 1: VALIDAÇÃO DAS CREDENCIAIS DA API META ---------- */
echo "<h2>Etapa 1 – Validação das Credenciais da API Meta</h2>\n";

// Inclui o autoloader do SDK do Facebook
require_once __DIR__ . '/facebook-sdk/Facebook/autoload.php';

$fb = null;
$instagram_business_account_id = '';
$facebook_page_id = FB_PAGE_ID; // Pega o ID da página do config.php

try {
    $fb = new Facebook\Facebook([
        'app_id' => FB_APP_ID,
        'app_secret' => FB_APP_SECRET,
        'default_graph_version' => 'v20.0',
    ]);
    echo "<span class='info'>Tentando validar o FB_PAGE_ID e FB_PAGE_ACCESS_TOKEN...</span>\n";

    $response = $fb->get('/' . FB_PAGE_ID . '?fields=id,name,instagram_business_account', FB_PAGE_ACCESS_TOKEN);
    $graphNode = $response->getGraphNode();
    echo "<span class='success'>Validação da API do Facebook CONCLUÍDA com sucesso!</span>\n";
    echo "<span class='info'>ID da Página: <b>" . htmlspecialchars($graphNode['id']) . "</b></span>\n";
    echo "<span class='info'>Nome da Página: <b>" . htmlspecialchars($graphNode['name']) . "</b></span>\n";

    if ($graphNode->offsetExists('instagram_business_account') && !empty($graphNode->offsetGet('instagram_business_account')['id'])) {
        $instagram_business_account_id = $graphNode->offsetGet('instagram_business_account')['id'];
        echo "<span class='success'>ID da Conta de Negócios do Instagram: <b>" . htmlspecialchars($instagram_business_account_id) . "</b></span>\n";
    } else {
        echo "<span class='warning'>AVISO: Esta página do Facebook NÃO está conectada a uma conta de Negócios do Instagram, ou a informação não pôde ser recuperada. A postagem no Instagram PODE NÃO funcionar.</span>\n";
    }

    echo "<span class='success'>Suas credenciais parecem estar OK.</span>\n";
} catch (Facebook\Exception\FacebookResponseException $e) {
    echo "<span class='error'>ERRO na validação da API do Facebook: " . htmlspecialchars($e->getMessage()) . "</span>\n";
    echo "<span class='warning'>Verifique suas credenciais e permissões.</span>\n";
    echo "</pre></body></html>";
    exit;
} catch (Facebook\Exception\FacebookSDKException $e) {
    echo "<span class='error'>ERRO do SDK do Facebook na validação: " . htmlspecialchars($e->getMessage()) . "</span>\n";
    echo "</pre></body></html>";
    exit;
} catch (Exception $e) {
    echo "<span class='error'>ERRO Geral na validação: " . htmlspecialchars($e->getMessage()) . "</span>\n";
    echo "</pre></body></html>";
    exit;
}

// Mensagem sobre o modo de depuração
if (!defined('ENABLE_ENGAGEMENT_REPLYING') || !ENABLE_ENGAGEMENT_REPLYING) {
    echo "\n<span class='warning'>ATENÇÃO: O script está em MODO DEBUG (ENABLE_ENGAGEMENT_REPLYING = false). NENHUMA resposta real será enviada para o Facebook ou Instagram.</span>\n";
    echo "<span class='warning'>  Isso é útil para testar a coleta de comentários e a lógica de verificação.</span>\n";
    echo "<span class='warning'>  Para ativar o envio de respostas reais, mude 'ENABLE_ENGAGEMENT_REPLYING' para 'true' no 'config.php'.</span>\n";
} else {
    echo "\n<span class='success'>MODO DE PRODUÇÃO ATIVO: Respostas reais serão enviadas.</span>\n";
}
echo "<div class='separator'></div>\n";

/* ---------- ETAPA 2: ENGAJAMENTO DE COMENTÁRIOS NO FACEBOOK ---------- */
echo "<h2>Etapa 2 – Engajamento no Facebook</h2>\n";

if ($conn === NULL) {
    echo "<span class='warning'>Conexão com o banco de dados falhou. Não é possível buscar posts do Facebook.</span>\n";
} else {
    try {
        $stmt = $conn->prepare("SELECT facebook_post_id FROM posts WHERE facebook_post_id IS NOT NULL AND facebook_post_id != ''");
        $stmt->execute();
        $result = $stmt->get_result();
        $facebook_posts = $result->fetch_all(MYSQLI_ASSOC);
        $stmt->close();

        if (empty($facebook_posts)) {
            echo "<span class='info'>Nenhum post do Facebook encontrado no banco de dados para processar.</span>\n";
        } else {
            echo "<span class='info'>Encontrados " . count($facebook_posts) . " posts do Facebook para verificar.</span>\n";
            flush();

            foreach ($facebook_posts as $post) {
                $post_id = $post['facebook_post_id'];
                echo "\n<span class='info'>Processando post do Facebook: <b>" . htmlspecialchars($post_id) . "</b></span>\n";
                flush();

                try {
                    // Obtém comentários e suas respostas para verificar se a página já respondeu
                    $comments_response = $fb->get(
                        '/' . $post_id . '/comments?fields=id,from,message,comments.limit(10){id,from,message,parent}&limit=100', // Pega 100 comentários, e 10 respostas para cada comentário
                        FB_PAGE_ACCESS_TOKEN
                    );
                    $comments_edge = $comments_response->getGraphEdge();

                    $comments_found_for_post = false;
                    if ($comments_edge->count() > 0) {
                        $comments_found_for_post = true;
                    }

                    foreach ($comments_edge as $comment) {
                        $comment_id = $comment['id'];
                        $comment_author = $comment['from']['name'];
                        $comment_message = $comment['message'];

                        echo "<div class='comment-box'>\n";
                        echo "<p><span class='author'>Comentário de " . htmlspecialchars($comment_author) . ":</span> <span class='message'>" . htmlspecialchars($comment_message) . "</span></p>\n";
                        echo "<p><span class='info'>ID do Comentário: " . htmlspecialchars($comment_id) . "</span></p>\n";
                        flush();

                        // 1. Verifica se é um comentário da própria página
                        // CORREÇÃO AQUI: Use ->asArray() para garantir que $comment['from'] seja um array
                        if (isPageComment($comment['from']->asArray(), $facebook_page_id)) {
                            echo "<span class='warning'>  → É um comentário da própria página. Ignorando.</span>\n";
                            echo "</div>\n";
                            continue;
                        }

                        // 2. Verifica se este script já respondeu a este comentário (pela tabela replied_comments)
                        if (isCommentReplied($conn, 'facebook', $comment_id)) {
                            echo "<span class='warning'>  → Já respondido por este script anteriormente. Ignorando.</span>\n";
                            echo "</div>\n";
                            continue;
                        }

                        // 3. Verifica se a própria página já respondeu a este comentário diretamente (buscando nas respostas do comentário)
                        $already_replied_by_page_api = false;
                        if ($comment->offsetExists('comments')) {
                            foreach ($comment['comments'] as $reply) {
                                // CORREÇÃO AQUI: Use ->asArray() para garantir que $reply['from'] seja um array
                                if (isPageComment($reply['from']->asArray(), $facebook_page_id)) {
                                    $already_replied_by_page_api = true;
                                    echo "<span class='warning'>  → A página JÁ RESPONDEU diretamente a este comentário (via API). Ignorando.</span>\n";
                                    // Marcar como respondido para este script também, para não tentar de novo
                                    if ($conn && (!defined('ENABLE_ENGAGEMENT_REPLYING') || ENABLE_ENGAGEMENT_REPLYING)) {
                                        markCommentAsReplied($conn, 'facebook', $post_id, $comment_id);
                                    }
                                    break;
                                }
                            }
                        }

                        if ($already_replied_by_page_api) {
                            echo "</div>\n";
                            continue;
                        }

                        // Se chegou aqui, é um comentário de usuário e não foi respondido por este script ou pela própria página
                        $reply_message = getRandomReplyPhrase('facebook'); // Passa a plataforma
                        echo "<span class='info'>  → Resposta potencial: <b>" . htmlspecialchars($reply_message) . "</b></span>\n";
                        flush();

                        if (!defined('ENABLE_ENGAGEMENT_REPLYING') || ENABLE_ENGAGEMENT_REPLYING) {
                            try {
                                $reply_response = $fb->post(
                                    '/' . $comment_id . '/comments',
                                    ['message' => $reply_message],
                                    FB_PAGE_ACCESS_TOKEN
                                );
                                $reply_node = $reply_response->getGraphNode();
                                echo "<span class='success'>  → Resposta publicada! ID da Resposta: " . htmlspecialchars($reply_node['id']) . "</span>\n";
                                // Apenas marca como respondido se a postagem real foi bem-sucedida
                                markCommentAsReplied($conn, 'facebook', $post_id, $comment_id);
                            } catch (Facebook\Exception\FacebookResponseException $e) {
                                echo "<span class='error'>  → ERRO ao responder comentário (API): " . htmlspecialchars($e->getMessage()) . " (Código: " . htmlspecialchars($e->getCode()) . ")</span>\n";
                                if ($e->getCode() == 10 || $e->getCode() == 200) { // Common codes for permission or invalid object
                                    echo "<span class='warning'>    Verifique as permissões do token ou se o comentário ainda existe/é válido.</span>\n";
                                }
                            } catch (Facebook\Exception\FacebookSDKException $e) {
                                echo "<span class='error'>  → ERRO do SDK ao responder comentário: " . htmlspecialchars($e->getMessage()) . "</span>\n";
                            }
                        } else {
                            echo "<span class='info simulated-message'>  → Modo DEBUG: Simulação da resposta: \"" . htmlspecialchars($reply_message) . "\"</span>\n";
                            echo "<span class='info'>  → Nenhuma resposta real enviada.</span>\n";
                        }
                        echo "</div>\n";
                        countdown(rand(2, 5)); // Pequena pausa após cada 'tentativa' de resposta
                    }

                    if (!$comments_found_for_post) {
                        echo "<span class='info'>  → Nenhum comentário encontrado para o post " . htmlspecialchars($post_id) . ".</span>\n";
                    }

                } catch (Facebook\Exception\FacebookResponseException $e) {
                    echo "<span class='error'>ERRO ao buscar comentários para o post " . htmlspecialchars($post_id) . ": " . htmlspecialchars($e->getMessage()) . " (Código: " . htmlspecialchars($e->getCode()) . ")</span>\n";
                    if ($e->getCode() == 100) { // Graph API Invalid Parameter
                        echo "<span class='warning'>    Pode ser um Post ID inválido ou privado.</span>\n";
                    }
                } catch (Facebook\Exception\FacebookSDKException $e) {
                    echo "<span class='error'>ERRO do SDK ao buscar comentários para o post " . htmlspecialchars($post_id) . ": " . htmlspecialchars($e->getMessage()) . "</span>\n";
                }
                countdown(rand(5, 10)); // Pausa entre posts para evitar limites de taxa
            }
            echo "\n<span class='success'>Verificação de posts do Facebook concluída.</span>\n";
        }
    } catch (Exception $e) {
        echo "<span class='error'>ERRO Geral na Etapa 2 (Facebook): " . htmlspecialchars($e->getMessage()) . "</span>\n";
    }
}

echo "<div class='separator'></div>\n";

/* ---------- ETAPA 3: ENGAJAMENTO DE COMENTÁRIOS NO INSTAGRAM ---------- */
echo "<h2>Etapa 3 – Engajamento no Instagram</h2>\n";

if (empty($instagram_business_account_id)) {
    echo "<span class='warning'>ID da conta de negócios do Instagram não disponível. Não é possível processar posts do Instagram.</span>\n";
} elseif ($conn === NULL) {
    echo "<span class='warning'>Conexão com o banco de dados falhou. Não é possível buscar posts do Instagram.</span>\n";
} else {
    try {
        $stmt = $conn->prepare("SELECT instagram_post_id FROM posts WHERE instagram_post_id IS NOT NULL AND instagram_post_id != ''");
        $stmt->execute();
        $result = $stmt->get_result();
        $instagram_posts = $result->fetch_all(MYSQLI_ASSOC);
        $stmt->close();

        if (empty($instagram_posts)) {
            echo "<span class='info'>Nenhum post do Instagram encontrado no banco de dados para processar.</span>\n";
        } else {
            echo "<span class='info'>Encontrados " . count($instagram_posts) . " posts do Instagram para verificar.</span>\n";
            flush();

            foreach ($instagram_posts as $post) {
                $post_id = $post['instagram_post_id']; // No Instagram, 'post_id' é na verdade o 'media_id'
                echo "\n<span class='info'>Processando post do Instagram (Media ID): <b>" . htmlspecialchars($post_id) . "</b></span>\n";
                flush();

                try {
                    // Para o Instagram, os comentários são buscados em /media_id/comments
                    $comments_response = $fb->get(
                        '/' . $post_id . '/comments?fields=id,from,text,replies.limit(10){id,from,text}&limit=100',
                        FB_PAGE_ACCESS_TOKEN
                    );
                    $comments_edge = $comments_response->getGraphEdge();

                    $comments_found_for_post = false;
                    if ($comments_edge->count() > 0) {
                        $comments_found_for_post = true;
                    }

                    foreach ($comments_edge as $comment) {
                        $comment_id = $comment['id'];
                        $comment_author = $comment['from']['username']; // Instagram usa 'username'
                        $comment_message = $comment['text']; // Instagram usa 'text'

                        echo "<div class='comment-box'>\n";
                        echo "<p><span class='author'>Comentário de " . htmlspecialchars($comment_author) . ":</span> <span class='message'>" . htmlspecialchars($comment_message) . "</span></p>\n";
                        echo "<p><span class='info'>ID do Comentário: " . htmlspecialchars($comment_id) . "</span></p>\n";
                        flush();

                        // 1. Instagram API não retorna o 'from' da página para comentários diretos no media.
                        //    A melhor forma é verificar se já há uma resposta da página ou se este script já respondeu.

                        // 2. Verifica se este script já respondeu a este comentário (pela tabela replied_comments)
                        if (isCommentReplied($conn, 'instagram', $comment_id)) {
                            echo "<span class='warning'>  → Já respondido por este script anteriormente. Ignorando.</span>\n";
                            echo "</div>\n";
                            continue;
                        }

                        // 3. Verifica se a própria página já respondeu a este comentário diretamente (buscando nas respostas do comentário)
                        $already_replied_by_page_api = false;
                        if ($comment->offsetExists('replies')) { // Instagram usa 'replies' para respostas a comentários
                            foreach ($comment['replies'] as $reply) {
                                // Para Instagram, o 'from' de uma resposta será o ID da conta de negócios do Instagram
                                // CORREÇÃO AQUI: Use ->asArray() para garantir que $reply['from'] seja um array
                                if (isPageComment($reply['from']->asArray(), $instagram_business_account_id)) {
                                    $already_replied_by_page_api = true;
                                    echo "<span class='warning'>  → A página JÁ RESPONDEU diretamente a este comentário (via API). Ignorando.</span>\n";
                                    // Marcar como respondido para este script também, para não tentar de novo
                                    if ($conn && (!defined('ENABLE_ENGAGEMENT_REPLYING') || ENABLE_ENGAGEMENT_REPLYING)) {
                                        markCommentAsReplied($conn, 'instagram', $post_id, $comment_id);
                                    }
                                    break;
                                }
                            }
                        }

                        if ($already_replied_by_page_api) {
                            echo "</div>\n";
                            continue;
                        }

                        // Se chegou aqui, é um comentário de usuário e não foi respondido por este script ou pela própria página
                        $reply_message = getRandomReplyPhrase('instagram'); // Passa a plataforma
                        echo "<span class='info'>  → Resposta potencial: <b>" . htmlspecialchars($reply_message) . "</b></span>\n";
                        flush();

                        if (!defined('ENABLE_ENGAGEMENT_REPLYING') || ENABLE_ENGAGEMENT_REPLYING) {
                            try {
                                // Para Instagram, as respostas são feitas para /comment_id/replies
                                $reply_response = $fb->post(
                                    '/' . $comment_id . '/replies',
                                    ['message' => $reply_message],
                                    FB_PAGE_ACCESS_TOKEN
                                );
                                $reply_node = $reply_response->getGraphNode();
                                echo "<span class='success'>  → Resposta publicada! ID da Resposta: " . htmlspecialchars($reply_node['id']) . "</span>\n";
                                // Apenas marca como respondido se a postagem real foi bem-sucedida
                                markCommentAsReplied($conn, 'instagram', $post_id, $comment_id);
                            } catch (Facebook\Exception\FacebookResponseException $e) {
                                echo "<span class='error'>  → ERRO ao responder comentário (API): " . htmlspecialchars($e->getMessage()) . " (Código: " . htmlspecialchars($e->getCode()) . ")</span>\n";
                                if ($e->getCode() == 10 || $e->getCode() == 200 || $e->getCode() == 190) { // Permissão negada, objeto inválido, token inválido
                                    echo "<span class='warning'>    Verifique as permissões do token ou se o comentário ainda existe/é válido no Instagram.</span>\n";
                                }
                            } catch (Facebook\Exception\FacebookSDKException $e) {
                                echo "<span class='error'>  → ERRO do SDK ao responder comentário: " . htmlspecialchars($e->getMessage()) . "</span>\n";
                            }
                        } else {
                            echo "<span class='info simulated-message'>  → Modo DEBUG: Simulação da resposta: \"" . htmlspecialchars($reply_message) . "\"</span>\n";
                            echo "<span class='info'>  → Nenhuma resposta real enviada.</span>\n";
                        }
                        echo "</div>\n";
                        countdown(rand(2, 5)); // Pequena pausa após cada 'tentativa' de resposta
                    }

                    if (!$comments_found_for_post) {
                        echo "<span class='info'>  → Nenhum comentário encontrado para o post " . htmlspecialchars($post_id) . ".</span>\n";
                    }

                } catch (Facebook\Exception\FacebookResponseException $e) {
                    echo "<span class='error'>ERRO ao buscar comentários para o post " . htmlspecialchars($post_id) . ": " . htmlspecialchars($e->getMessage()) . " (Código: " . htmlspecialchars($e->getCode()) . ")</span>\n";
                    if ($e->getCode() == 100) { // Invalid parameter
                        echo "<span class='warning'>    Pode ser um Media ID inválido ou privado.</span>\n";
                    }
                } catch (Facebook\Exception\FacebookSDKException $e) {
                    echo "<span class='error'>ERRO do SDK ao buscar comentários para o post " . htmlspecialchars($post_id) . ": " . htmlspecialchars($e->getMessage()) . "</span>\n";
                }
                countdown(rand(5, 10)); // Pausa entre posts para evitar limites de taxa
            }
            echo "\n<span class='success'>Verificação de posts do Instagram concluída.</span>\n";
        }
    } catch (Exception $e) {
        echo "<span class='error'>ERRO Geral na Etapa 3 (Instagram): " . htmlspecialchars($e->getMessage()) . "</span>\n";
    }
}

echo "<div class='separator'></div>\n";

/* ---------- FINALIZAÇÃO ---------- */
if ($conn) {
    $conn->close();
    echo "<span class='info'>Conexão com o banco de dados MySQL encerrada.</span>\n";
}
echo "<span class='info'>Processo de engajamento finalizado.</span>\n";
echo '</pre></body></html>';

?>