<?php
/**
 * social_media_poster.php — Versão atualizada com Facebook e Instagram, Debug e Preview
 * Atualizado: 22/06/2025
 * -----------------------------------------------------------
 * Este script automatiza o scraping de dados de releases e a postagem no Facebook e Instagram.
 * Inclui depuração para identificar o status das operações e um preview do post.
 */

/* ---------- CONFIGURAÇÃO DE BUFFER ---------- */
// Desativa o buffering de saída para exibir o progresso em tempo real
while (ob_get_level() > 0) { ob_end_clean(); }
ob_implicit_flush(true);
header('X-Accel-Buffering: no');
if (function_exists('apache_setenv')) { apache_setenv('no-gzip', '1'); }
header('Content-Type: text/html; charset=utf-8');

// Define o caminho para o SDK do Facebook. Certifique-se de que ele está correto.
$sdkPath = __DIR__ . '/facebook-sdk'; // Ajuste se o diretório for diferente

// Autoload para carregar as classes do SDK do Facebook
// CORREÇÃO: Revertido para usar $sdkPath corretamente
spl_autoload_register(function ($class) use ($sdkPath) {
    $file = $sdkPath . DIRECTORY_SEPARATOR . str_replace('\\', DIRECTORY_SEPARATOR, $class) . '.php';
    if (is_file($file)) {
        require_once $file;
    }
});

// ------------------------ Configurações ------------------------
// Preencha com os dados do seu aplicativo e página do Facebook/Instagram.
// Certifique-se de que o FB_USER_LONG_TOKEN é um token de longa duração e tem as permissões corretas (publish_to_groups, pages_show_list, instagram_basic, instagram_content_publish, pages_read_engagement).
const FB_APP_ID             = '570218186143869';
const FB_APP_SECRET         = 'c8a0c8b5b6015ae66eb99577dde4f8de';
const FB_PAGE_ID            = '107018730989128'; // ID da sua página do Facebook
const FB_USER_LONG_TOKEN    = 'EAAIGnERtZBH0BO4L0c1JifYEHvBnyiIYXH7E03oPuAPQZANfcjOeM5T2hZBfaqDW0Kmt1qAP22a9cG2sxX4TThLMyfbZBp4s3721ZAHsWTd1sbZAmJLMKu4OBitvkCZBphGGQUPUW86ccnBGj1o0pu0ZBrZC0xZARrEfERFcFbXfed4ChuFRsrfoMZBZASrYKg03lyemLTCZAua5P5Ykk';
const INSTAGRAM_BUSINESS_ID = '17841433749115146'; // ID da sua conta comercial do Instagram
const MAX_RELEASE_NUMBER    = 35;
const BASE_URL              = 'https://push.fm/fl/fht';

// Modo de depuração: true para ativar, false para desativar.
const DEBUG_MODE = false; // Alterne para false em produção.

// ---------------------- Funções Auxiliares de Log -------------------------

/**
 * Exibe mensagens de depuração se o DEBUG_MODE estiver ativado.
 * @param string $message A mensagem a ser exibida.
 */
function debugLog(string $message): void
{
    if (DEBUG_MODE) {
        echo "<span class='debug'><i class='fa-solid fa-bug'></i> DEBUG: " . htmlspecialchars($message) . "</span>\n";
    }
}

/**
 * Exibe mensagens de sucesso.
 * @param string $message A mensagem a ser exibida.
 */
function successLog(string $message): void
{
    echo "<span class='success'><i class='fa-solid fa-check-circle'></i> " . htmlspecialchars($message) . "</span>\n";
}

/**
 * Exibe mensagens de erro.
 * @param string $message A mensagem a ser exibida.
 */
function errorLog(string $message): void
{
    echo "<span class='error'><i class='fa-solid fa-times-circle'></i> ERRO: " . htmlspecialchars($message) . "</span>\n";
}

/**
 * Exibe mensagens de informação.
 * @param string $message A mensagem a ser exibida.
 */
function infoLog(string $message): void
{
    echo "<span class='info'><i class='fa-solid fa-info-circle'></i> " . htmlspecialchars($message) . "</span>\n";
}

/**
 * Exibe mensagens de aviso.
 * @param string $message A mensagem a ser exibida.
 */
function warningLog(string $message): void
{
    echo "<span class='warning'><i class='fa-solid fa-exclamation-triangle'></i> " . htmlspecialchars($message) . "</span>\n";
}

/**
 * Realiza uma requisição à API Graph do Facebook.
 *
 * @param string $method Método HTTP (GET ou POST).
 * @param string $endpoint Endpoint da API Graph.
 * @param string $token Token de acesso (usuário ou página).
 * @param array $params Parâmetros da requisição.
 * @return array Corpo da resposta decodificado.
 */
function fbRequest(string $method, string $endpoint, string $token, array $params = []): array
{
    debugLog("Fazendo requisição {$method} para: {$endpoint}");
    $fb = new Facebook\Facebook([
        'app_id'                => FB_APP_ID,
        'app_secret'            => FB_APP_SECRET,
        'default_graph_version' => 'v19.0', // Versão 19.0 da API Graph
    ]);

    $fullEndpoint = $endpoint;
    if ($method === 'GET' && $params) {
        $fullEndpoint .= '?' . http_build_query($params);
    }

    try {
        $response = ($method === 'GET')
            ? $fb->get($fullEndpoint, $token)
            : $fb->post($fullEndpoint, $params, $token);
        return $response->getDecodedBody();
    } catch (Facebook\Exception\FacebookResponseException $e) {
        // Quando Graph retorna um erro
        debugLog('Graph returned an error: ' . $e->getMessage());
        throw new RuntimeException('Erro na API do Facebook: ' . $e->getMessage());
    } catch (Facebook\Exception\FacebookSDKException $e) {
        // Quando ocorre um erro com o SDK
        debugLog('Facebook SDK returned an error: ' . $e->getMessage());
        throw new RuntimeException('Erro no SDK do Facebook: ' . $e->getMessage());
    }
}

/**
 * Obtém o token de acesso da página usando o token de usuário de longa duração.
 *
 * @return string O token de acesso da página.
 * @throws RuntimeException Se a página não for encontrada ou não houver páginas visíveis.
 */
function getPageAccessToken(): string
{
    debugLog("Chamando /me/accounts para obter páginas visíveis...");
    $data = fbRequest('GET', '/me/accounts', FB_USER_LONG_TOKEN, [
        'fields' => 'id,name,access_token'
    ]);

    $pages = $data['data'] ?? [];
    debugLog("/me/accounts retornou " . count($pages) . " página(s).");

    if (!$pages) {
        throw new RuntimeException('Nenhuma página retornada pelo token de acesso do usuário. Verifique as permissões.');
    }

    foreach ($pages as $p) {
        if ($p['id'] === FB_PAGE_ID) {
            debugLog('Página encontrada: ' . htmlspecialchars($p['name']) . ' (ID ' . $p['id'] . ')');
            return $p['access_token'];
        }
    }
    throw new RuntimeException('Sua página ID ' . FB_PAGE_ID . ' não foi encontrada entre as páginas visíveis pelo token. Verifique se o ID está correto e as permissões.');
}

/**
 * Valida o token de acesso da página tentando obter o nome da página.
 *
 * @param string $pageToken O token de acesso da página a ser validado.
 */
function validatePageToken(string $pageToken): void
{
    $info = fbRequest('GET', '/' . FB_PAGE_ID, $pageToken, ['fields' => 'name']);
    debugLog('Token válido para a página: ' . htmlspecialchars($info['name']));
}

/**
 * Realiza o scraping dos dados de uma URL de release usando cURL com User-Agent.
 *
 * @param string $url A URL da página de release para fazer o scraping.
 * @return array Um array contendo 'imagem_url', 'descricao' e 'release_url', ou 'erro' em caso de falha.
 */
function scrapeReleaseData(string $url): array
{
    debugLog("Iniciando scraping de {$url} (usando cURL)...");

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36');

    $html = curl_exec($ch);

    if (curl_errno($ch)) {
        $error_msg = curl_error($ch);
        curl_close($ch);
        debugLog("Erro cURL: " . htmlspecialchars($error_msg));
        return ['erro' => 'Falha ao baixar (cURL error): ' . $error_msg];
    }

    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($http_code >= 400) {
        debugLog("Erro HTTP ao baixar: Status " . $http_code);
        return ['erro' => 'Falha ao baixar (HTTP Status ' . $http_code . ')'];
    }

    if (empty($html)) {
        debugLog("Conteúdo HTML vazio retornado.");
        return ['erro' => 'Conteúdo HTML vazio retornado'];
    }

    $doc = new DOMDocument();
    @$doc->loadHTML($html);
    $xp = new DOMXPath($doc);

    $img = $xp->evaluate('string(//meta[@property="og:image"]/@content)');
    $desc = $xp->evaluate('string(//meta[@property="og:description"]/@content)');

    if (!$img || !$desc) {
        debugLog("Meta tag og:image ou og:description ausente ou vazia no HTML.");
        return ['erro' => 'og:image ou og:description ausente ou não parseado do HTML'];
    }

    debugLog("Dados do scraping extraídos com sucesso.");
    return [
        'imagem_url'  => $img,
        'descricao'   => $desc,
        'release_url' => $url,
    ];
}

/**
 * Constrói a mensagem padrão para as postagens baseada na plataforma.
 *
 * @param array $d Dados da release (release_url).
 * @param string $platform A plataforma de destino ('facebook' ou 'instagram').
 * @return array Um array com a mensagem e as hashtags.
 */
function buildPostMessage(array $d, string $platform): array
{
    $message = "💥 House music lovers, press play and feel the vibe!\n\n";
    $message .= "One of our favorite cuts from the catalog — now just a click away on all major platforms!\n\n";

    if ($platform === 'facebook') {
        $message .= "Streaming & download 👉 " . $d['release_url'] . "\n\n";
        $message .= "⚠️We are accepting demos, sent yours here 👉🔗https://finesthousetraxx.com";
    } else { // Instagram
        $message .= "Streaming & download 👉 " . $d['release_url'] . " or hit our bio link🎯\n\n";
        $message .= "⚠️We are accepting demos, sent yours, 🔗 is in our bio, check it!";
    }

    // Esta linha deve estar no final da mensagem principal, antes das hashtags.
    $message .= "\n\nFinest House Traxx: House Sweet House💛";

    $hashtags = "#HouseMusic #DeepHouse #JackinHouse #SoulfulHouse #AfroHouse #NuDisco #FunkyHouse #UndergroundHouse #ClassicHouse #HouseHeads #HouseVibes #HouseGroove #HouseCommunity #DJLife #RealHouseMusic #DancefloorVibes #ClubReady #HouseIsAFeeling #HouseCulture #Beatport #Traxsource #SpotifyHouse #iTunesMusic #JunoDownload #SupportHouseMusic";

    return ['message' => $message, 'hashtags' => $hashtags];
}

/**
 * Publica uma foto com uma mensagem na página do Facebook.
 *
 * @param array $d Dados da release (imagem_url, release_url).
 * @param string $token O token de acesso da página.
 */
function postToFacebook(array $d, string $token): void
{
    debugLog("Iniciando postagem no Facebook...");
    $post = buildPostMessage($d, 'facebook'); // Passa a plataforma 'facebook'
    $fullMessage = $post['message'] . "\n\n" . $post['hashtags'];

    $res = fbRequest('POST', '/' . FB_PAGE_ID . '/photos', $token, [
        'message' => $fullMessage,
        'url'     => $d['imagem_url'],
    ]);
    debugLog('Postagem no Facebook criada com ID: ' . htmlspecialchars($res['id']));
    successLog("Postado com sucesso no Facebook! ID: " . htmlspecialchars($res['id']));
}

/**
 * Publica uma imagem no Instagram Business Account.
 *
 * @param array $d Dados da release (imagem_url, release_url).
 * @param string $token O token de acesso da página.
 */
function postToInstagram(array $d, string $token): void
{
    debugLog("Iniciando postagem no Instagram...");
    $post = buildPostMessage($d, 'instagram'); // Passa a plataforma 'instagram'
    $fullCaption = $post['message'] . "\n\n" . $post['hashtags'];

    // 1. Criar o contêiner de mídia
    debugLog("Criando contêiner de mídia para Instagram...");
    $containerRes = fbRequest('POST', '/' . INSTAGRAM_BUSINESS_ID . '/media', $token, [
        'image_url' => $d['imagem_url'],
        'caption'   => $fullCaption,
    ]);
    $containerId = $containerRes['id'] ?? null;

    if (!$containerId) {
        throw new RuntimeException('Falha ao criar o contêiner de mídia para o Instagram.');
    }
    debugLog("Contêiner de mídia do Instagram criado com ID: " . $containerId);

    // 2. Publicar a mídia
    debugLog("Publicando mídia no Instagram...");
    $publishRes = fbRequest('POST', '/' . INSTAGRAM_BUSINESS_ID . '/media_publish', $token, [
        'creation_id' => $containerId,
    ]);
    $mediaId = $publishRes['id'] ?? null;

    if (!$mediaId) {
        throw new RuntimeException('Falha ao publicar a mídia no Instagram.');
    }
    debugLog("Mídia do Instagram publicada com ID: " . $mediaId);
    successLog("Postado com sucesso no Instagram! ID: " . $mediaId);
}

/**
 * Exibe um preview do post para Facebook e Instagram.
 * @param array $data Os dados da release (imagem_url, release_url).
 */
function showPostPreview(array $data): void
{
    // Constrói mensagens específicas para cada plataforma para o preview
    $facebookPost = buildPostMessage($data, 'facebook');
    $instagramPost = buildPostMessage($data, 'instagram');

    $fullFacebookMessage = $facebookPost['message'] . "\n\n" . $facebookPost['hashtags'];
    $fullInstagramCaption = $instagramPost['message'] . "\n\n" . $instagramPost['hashtags'];

    echo "\n<h2 class='preview-title'><i class='fa-solid fa-eye'></i> Preview das Postagens</h2>\n";
    echo '<div class="flex-container">';

    // Preview do Facebook
    echo '<div class="post-card facebook-card">
                <h3><i class="fa-brands fa-facebook"></i> Facebook Post</h3>
                <img src="' . htmlspecialchars($data['imagem_url']) . '" alt="Preview da imagem">
                <p>' . htmlspecialchars($fullFacebookMessage) . '</p>
              </div>';

    // Preview do Instagram
    echo '<div class="post-card instagram-card">
                <h3><i class="fa-brands fa-instagram"></i> Instagram Post</h3>
                <img src="' . htmlspecialchars($data['imagem_url']) . '" alt="Preview da imagem">
                <p>' . htmlspecialchars($fullInstagramCaption) . '</p>
              </div>';

    echo '</div>'; // Fim do flex container
    echo "\n<hr>\n";
}


// ------------------- INÍCIO DA EXECUÇÃO DO SCRIPT -------------------------
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Andamento do Scraper Traxsource</title>
    <link href="https://fonts.googleapis.com/css2?family=Bebas+Neue&family=Open+Sans:wght@400;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" integrity="sha512-SnH5WK+bZxgPHs44uWIX+LLJAJ9/2PkPKZ5QiAj6Ta86w+fsb2TkcmfRyVX3pBnMFcV7oQPJkl9QevSCWr3W6A==" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <style>
        body {
            background: #1a1a1a; /* Cinza escuro suave */
            color: #FFFF00; /* Amarelo puro para o texto principal */
            font-family: 'Open Sans', sans-serif;
            font-weight: 400; /* Garante o peso da fonte Open Sans */
            margin: 0;
            padding: 20px;
            line-height: 1.6;
            white-space: pre-wrap; /* Mantém a quebra de linha */
        }
        h1 {
            font-family: 'Bebas Neue', cursive;
            font-size: 3em; /* Aumentado para mais impacto */
            color: #FFFF00;
            margin-top: 30px;
            margin-bottom: 20px;
            text-align: center;
            letter-spacing: 2px; /* Adiciona espaçamento entre letras */
            text-shadow: 0 0 5px rgba(255, 255, 0, 0.5); /* Sutil brilho amarelo */
        }
        h2 {
            font-family: 'Bebas Neue', cursive;
            font-size: 2.2em;
            color: #FFFF00;
            margin-top: 30px;
            margin-bottom: 15px;
            border-bottom: 2px solid rgba(255, 255, 0, 0.3); /* Linha amarela sutil */
            padding-bottom: 10px;
            text-shadow: 0 0 3px rgba(255, 255, 0, 0.3);
        }
        h1 i, h2 i, h3 i {
            margin-right: 10px; /* Espaço entre ícone e texto */
        }
        span.success {
            color: #00FF00; /* Verde vibrante */
            font-weight: bold;
            display: block;
        }
        span.error {
            color: #FF0000; /* Vermelho vibrante */
            font-weight: bold;
            display: block;
        }
        span.warning {
            color: #FFA500; /* Laranja */
            font-weight: bold;
            display: block;
        }
        span.info {
            color: #40E0D0; /* Turquesa/Ciano vibrante */
            display: block;
        }
        span.debug {
            color: #AAAAAA; /* Cinza claro para debug */
            font-size: 0.9em;
            display: block;
        }
        a {
            color: #40E0D0; /* Ciano para links */
            text-decoration: none;
            transition: color 0.3s ease;
        }
        a:hover {
            color: #00CED1; /* Ciano mais escuro no hover */
            text-decoration: underline;
        }
        hr {
            border: 0;
            height: 1px;
            background-image: linear-gradient(to right, rgba(255, 255, 0, 0), rgba(255, 255, 0, 0.75), rgba(255, 255, 0, 0));
            margin: 20px 0;
        }
        /* Estilos para o preview */
        .flex-container {
            display: flex;
            justify-content: center; /* Centraliza os cartões */
            flex-wrap: wrap;
            gap: 20px; /* Espaçamento entre os cartões */
            margin-top: 20px;
        }
        .post-card {
            width: 45%; /* Reduzido ligeiramente para garantir gap */
            max-width: 450px; /* Largura máxima para cartões */
            min-width: 320px;
            padding: 20px;
            border: 2px solid #FFFF00; /* Borda amarela mais grossa */
            border-radius: 10px; /* Bordas mais arredondadas */
            background-color: #282828; /* Fundo ligeiramente mais claro que o body */
            box-shadow: 0 5px 15px rgba(0,0,0,0.7); /* Sombra mais forte */
            color: #FFFF00; /* Texto amarelo dentro dos cartões */
            box-sizing: border-box; /* Inclui padding e border na largura */
        }
        .post-card h3 {
            font-family: 'Bebas Neue', cursive;
            font-size: 2em;
            color: #FFFF00; /* Título do post card em amarelo */
            margin-bottom: 15px;
            border-bottom: 1px solid rgba(255, 255, 0, 0.2);
            padding-bottom: 5px;
        }
        .post-card img {
            max-width: 100%;
            height: auto;
            border-radius: 5px; /* Bordas arredondadas para imagens */
            margin-bottom: 15px;
            border: 1px solid rgba(255, 255, 0, 0.5); /* Borda sutil na imagem */
        }
        .post-card p {
            font-family: 'Open Sans', sans-serif;
            font-weight: 400; /* Garante o peso da fonte Open Sans */
            white-space: pre-wrap;
            font-size: 0.95em; /* Ligeiramente menor para compactar */
            line-height: 1.6;
            color: #FFFF00; /* Texto do post em amarelo */
        }
        /* Cores específicas para ícones de rede social nos cards */
        .facebook-card h3 i { color: #1877F2; } /* Azul Facebook */
        .instagram-card h3 i { 
            background: -webkit-linear-gradient(45deg, #f09433 0%,#e6683c 25%,#dc2743 50%,#cc2366 75%,#bc1888 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }

    </style>
</head>
<body>
    <?php
    echo "<h1><i class='fa-solid fa-robot'></i> Automatizador de Postagens em Redes Sociais</h1>\n";
    echo "<hr>\n";

    if (DEBUG_MODE) {
        warningLog("Modo de Depuração (DEBUG_MODE) está ATIVO. Mensagens detalhadas serão exibidas. Desative `DEBUG_MODE` para produção e para as postagens serem efetuadas.");
    }
    echo "<hr>\n";

    try {
        echo "<h2><i class='fa-solid fa-play-circle'></i> Iniciando Processo...</h2>\n";

        // 1. Obtém e valida o token de acesso da página
        infoLog("Verificando Token de Acesso da Página...");
        $pageToken = getPageAccessToken();
        validatePageToken($pageToken);
        successLog("Token de acesso da página validado com sucesso.");
        echo "<hr>\n";

        // 2. Gera uma URL de release aleatória
        // FORMATADO COMO H2 COM ÍCONE
        echo "<h2><i class='fa-solid fa-link'></i> Gerando URL da Release:</h2>\n";
        $randomNumber = mt_rand(1, MAX_RELEASE_NUMBER);
        $url = BASE_URL . sprintf('%03d', $randomNumber);
        infoLog("URL da release a ser raspada: <a href='" . htmlspecialchars($url) . "' target='_blank'>" . htmlspecialchars($url) . "</a>");
        echo "<hr>\n";

        // 3. Tenta fazer o scraping dos dados da release da URL
        // FORMATADO COMO H2 COM ÍCONE
        echo "<h2><i class='fa-solid fa-magnifying-glass'></i> Realizando Scraping dos Dados...</h2>\n";
        $data = scrapeReleaseData($url);
        if (isset($data['erro'])) {
            throw new RuntimeException($data['erro']);
        }
        successLog("Dados da release raspados com sucesso!");
        infoLog("Imagem: <a href='" . htmlspecialchars($data['imagem_url']) . "' target='_blank'>" . htmlspecialchars($data['imagem_url']) . "</a>");
        infoLog("Descrição (OG): " . htmlspecialchars($data['descricao']));
        echo "<hr>\n";

        // 4. Exibe o preview do post
        showPostPreview($data);
        echo "<hr>\n";

        // 5. Se o scraping foi bem-sucedido e DEBUG_MODE está desativado, posta a release
        if (!DEBUG_MODE) {
            infoLog("Publicando nas Redes Sociais:");
            postToFacebook($data, $pageToken);
            postToInstagram($data, $pageToken);
            successLog("Fluxo finalizado! Postagens realizadas com sucesso.");
        } else {
            warningLog("DEBUG_MODE está ATIVO: As postagens NÃO foram enviadas para as redes sociais. Desative o modo de depuração para publicar.");
        }

    } catch (Throwable $e) {
        // Captura e exibe qualquer exceção/erro que ocorra em qualquer etapa
        errorLog("Erro Crítico: " . htmlspecialchars($e->getMessage()));
        if (DEBUG_MODE) {
            errorLog("Arquivo: " . htmlspecialchars($e->getFile()));
            errorLog("Linha: " . htmlspecialchars($e->getLine()));
        }
    }
    echo "<hr>\n";
    infoLog("Processo concluído.");
    ?>
</body>
</html>